/*
Author: Nathaniel Huesler
Year: 2025
Copywrite Notice: LICENSE.txt 
*/

/*

This is the platform layer. It is responsible for creating the window, providing user input,
and proving procedures to access the file-system and display the screen contents. 

This is compiled seperately from the main program, and acts as a foundation. This file calls an entry function
defined in the main program, which it dynamically loads on startup. When the main program returns, this file
loops and repeats the same process until the user closes the window. 

*/

#include <float.h>
#include <stdio.h>
#include <math.h>
#include "windows.h"

#define PRAGMA_OPTIMISE on

#include "util.cpp"
#include "platform.cpp"
#include "input.h"

#include "platform-win.cpp"


typedef u32 (EntryFuncType)(
							int argc , char ** argv , u32 * running , 
							PlatformScreenBuffer screen_buffer , 
							InputState * input_state , f32 delta_time , 
							PlatformProcedures * procs , 
							void ** user_data
							);

EntryFuncType * load_dll_code(
							  char * filename , 
							  char * path , 
							  char * func_name , 
							  HMODULE * module_handle)
{
	
	if(*module_handle) FreeLibrary(*module_handle);
	
	char dll[MAX_PATH] = {};
	char active_dll[MAX_PATH] = {};
	char temp[MAX_PATH] = {};
	
	str_concat(dll , path , filename); // path/filename
	str_concat(temp , "active_" , filename); // active_<filename>
	str_concat(active_dll , path , temp); // path/active_<filename>
	CopyFile(dll , active_dll , FALSE);
	
	*module_handle = LoadLibraryA(dll);
	EntryFuncType * entry = (EntryFuncType*)GetProcAddress(*module_handle , func_name);
	return(entry);
}


f32 platform_delta_time(LARGE_INTEGER frame_start_time)
{
	LARGE_INTEGER frame_end_time = platform_get_time();
	f32 delta_time = platform_elapsed_time(frame_start_time , frame_end_time);
	return(delta_time);
}


void platform_update_screen(
							PlatformScreenBuffer * buffer , 
							BITMAPINFO * bitmap_info , 
							HDC device_context , 
							i32 client_width , 
							i32 client_height)
{
	
	//C:/Program Files (x86)/Windows Kits/8.1/include/um/wingdi.h
	int result = -1;
	//u32 map_mode = GetMapMode(device_context);
	// 0xFF000000 -> 0x000000FF
	//platform_clear_screen(*buffer , 0xFF000000);
	result = StretchDIBits(
						   device_context , 
						   0 , 0 , bitmap_info -> bmiHeader.biWidth , bitmap_info -> bmiHeader.biHeight , 
						   0 , 0 , bitmap_info -> bmiHeader.biWidth , bitmap_info -> bmiHeader.biHeight , 
						   buffer -> pixels , bitmap_info , 
						   DIB_RGB_COLORS , 
						   SRCCOPY
						   );
	
#if 0
	
	result = SetDIBitsToDevice(
							   device_context , 
							   0 , 0 , buffer -> stride , buffer -> height , 0 , 0 ,
							   0 , buffer -> height , buffer -> pixels , bitmap_info , 
							   DIB_RGB_COLORS 
							   //SRCCOPY
							   );
	
#else 
	
	
	
#endif
	
}

void platform_resize_screen_buffer(
								   PlatformScreenBuffer * buffer , 
								   BITMAPINFO * bitmap_info , 
								   u32 width , u32 height , u32 pixel_format)
{
	//CYCLE_TIMER(platform_resize_screen_buffer);
	
	if(buffer -> width == width && buffer -> height == height) return;
	if(buffer -> data != NULL)
	{
		VirtualFree(buffer -> data , 0 , MEM_RELEASE);
		buffer -> data = NULL;
	}
	
	u32 bytes_per_pixel = 4;
	u32 stride = width + (4 - (width % 4));
	u32 pixel_count = stride*(height+2);
	assert(!(stride%4));
	
	*bitmap_info = BITMAPINFO{};
	bitmap_info -> bmiHeader.biSize = sizeof(bitmap_info -> bmiHeader);
	bitmap_info -> bmiHeader.biWidth = stride;
	bitmap_info -> bmiHeader.biHeight = height;
	bitmap_info -> bmiHeader.biPlanes = 1;
	bitmap_info -> bmiHeader.biBitCount = (WORD)(bytes_per_pixel * 8);
	bitmap_info -> bmiHeader.biCompression = BI_RGB;
	
	buffer -> width = width;
	buffer -> height = height;
	buffer -> bytes_per_pixel = bytes_per_pixel;
	buffer -> stride = stride;
	
#if 0
	
	if(pixel_format == ARGB_PIXEL_FORMAT)
	{
		buffer -> red_mask = 	0x00FF0000;
		buffer -> green_mask = 	0x0000FF00;
		buffer -> blue_mask = 	0x000000FF;
		buffer -> alpha_mask = 	0xFF000000;
	}
	
	else if(pixel_format == RGBA_PIXEL_FORMAT)
	{
		buffer -> red_mask = 	0x000000FF;
		buffer -> green_mask = 	0x0000FF00;
		buffer -> blue_mask = 	0x00FF0000;
		buffer -> alpha_mask = 	0xFF000000;
	}
	
	else
	{
		assert_zero;
	}
	
	buffer -> red_shift = platform_bitscan_left(buffer -> red_mask);
	buffer -> green_shift = platform_bitscan_left(buffer -> green_mask);
	buffer -> blue_shift = platform_bitscan_left(buffer -> blue_mask);
	buffer -> alpha_shift = platform_bitscan_left(buffer -> alpha_mask);
	
#endif
	
	buffer -> data = (u8*)VirtualAlloc(0 , pixel_count*bytes_per_pixel , MEM_COMMIT , PAGE_READWRITE);
	buffer -> pixel_count = pixel_count;
	buffer -> pixels = (u8*)buffer -> data + buffer -> stride*buffer -> bytes_per_pixel;
	
}


void platform_resize_screen_buffer_to_window(
											 PlatformScreenBuffer * screen_buffer , HWND window_handle , BITMAPINFO * bitmap_info , u32 pixel_format)
{
	RECT client_rect;
	GetClientRect(window_handle , &client_rect);
	i32 client_width = client_rect.right - client_rect.left;
	i32 client_height = client_rect.bottom - client_rect.top;
	platform_resize_screen_buffer(screen_buffer , bitmap_info , client_width , client_height , pixel_format);
}

void platform_initialise_screen_buffer(PlatformScreenBuffer * screen_buffer , HWND window_handle , BITMAPINFO * bitmap_info , u32 pixel_format)
{
	
	UpdateWindow(window_handle);
	ShowWindow(window_handle , SW_SHOWDEFAULT);
	platform_resize_screen_buffer_to_window(screen_buffer , window_handle , bitmap_info , pixel_format);
}




void platform_set_key(KeyState * state , u32 action , u32 timestamp_ms)
{
	
	if(state -> action != action)
	{
		
		if(state -> timestamp.high == 0 && state -> timestamp.low == 0)
		{
			state -> timestamp.low = timestamp_ms;
		}
		
		if(state -> counter == 0) state -> counter = KEY_DURATION_CAPACITY;
		u32 index = (--state -> counter) % KEY_DURATION_CAPACITY;
		
		LARGE_INTEGER curr_time = platform_get_time();
		curr_time.LowPart = timestamp_ms;
		
		state -> state_durations[index] = 0;
		//state -> state_durations[index] = platform_delta_time(curr_time);
		state -> timestamp.low = timestamp_ms;
		state -> action = action;
		state -> transitions++;
	}
}


void platform_set_key(KeyState * state , u32 action , DoubleInt new_timestamp)
{
	
	if(state -> action != action)
	{
		
		if(state -> timestamp.high == 0 && state -> timestamp.low == 0)
		{
			state -> timestamp = new_timestamp;
		}
		
		if(state -> counter == 0) state -> counter = KEY_DURATION_CAPACITY;
		u32 index = (--state -> counter) % KEY_DURATION_CAPACITY;
		
		LARGE_INTEGER time = {};
		time.HighPart = new_timestamp.high;
		time.LowPart = new_timestamp.low;
		state -> state_durations[index] = 0;
		state -> timestamp = new_timestamp;
		state -> action = action;
		state -> transitions++;
	}
}

u32 platform_proccess_mouse_input(InputState * input_state , MSG * message , u32 client_height)
{
	
	UINT msg = message -> message;
	LPARAM l_param = message -> lParam;	
	WPARAM w_param = message -> wParam;
	MouseInput * mouse = &input_state -> mouse;
	u32 action = 0;
	u32 button = 0;
	u32 result = 1;
	
	u32 timestamp_ms = message -> time;
	
	if(msg == WM_MOUSEMOVE)
	{
		
		f32 new_x = (int)(l_param & 0x0000FFFF);
		f32 new_y = client_height - (int)((l_param >> 16) & 0x0000FFFF);
		
		mouse -> abs_pos_x = new_x;
		mouse -> abs_pos_y = new_y;
	}
	
	else if(msg == WM_MOUSEWHEEL)
	{
		f32 delta = (i16)((w_param >> 16) & 0x0000FFFF);
		mouse -> scroll_wheel = delta/120.0f;
	}
	
	else
	{
		switch(msg)
		{			
			case(WM_LBUTTONDOWN):
			{
				action = 1;
				button = MOUSE_PRIMARY;
				break;
			}
			
			case(WM_LBUTTONUP):
			{
				action = 0;
				button = MOUSE_PRIMARY;
				break;
			}
			
			case(WM_RBUTTONDOWN):
			{
				action = 1;
				button = MOUSE_SECONDARY;
				break;
			}
			
			case(WM_RBUTTONUP):
			{
				action = 0;
				button = MOUSE_SECONDARY;
				break;
			}
			
			case(WM_MBUTTONDOWN):
			{
				action = 1;
				button = MOUSE_MIDDLE;
				break;
			}
			
			case(WM_MBUTTONUP):
			{
				action = 0;
				button = MOUSE_MIDDLE;
				break;
			}
		}
		
		if(button)
		{
			platform_set_key(input_state -> keys + button , action , timestamp_ms);
		}
		
		else
		{
			result = 0;
		}
	}
	
	return(result);
}

bool platform_process_keyboard_input(InputState * input_state , MSG * message)
{
	bool result = 1;
	UINT msg = message -> message;
	WPARAM w_param = message -> wParam;
	u32 key = 0;
	u32 timestamp_ms = message -> time;
	
	u32 virtual_key_code = w_param;
	u32 scan_code = MapVirtualKeyA(virtual_key_code , MAPVK_VK_TO_VSC);
	
	if(msg == WM_KEYDOWN || msg == WM_KEYUP)
	{
		switch(virtual_key_code)
		{
			case('A'): key = KEY_A; break;
			case('B'): key = KEY_B; break;
			case('C'): key = KEY_C; break;
			case('D'): key = KEY_D; break;
			case('E'): key = KEY_E; break;
			case('F'): key = KEY_F; break;
			case('G'): key = KEY_G; break;
			case('H'): key = KEY_H; break;
			case('I'): key = KEY_I; break;
			case('J'): key = KEY_J; break;
			case('K'): key = KEY_K; break;
			case('L'): key = KEY_L; break;
			case('M'): key = KEY_M; break;
			case('N'): key = KEY_N; break;
			case('O'): key = KEY_O; break;
			case('P'): key = KEY_P; break;
			case('Q'): key = KEY_Q; break;
			case('R'): key = KEY_R; break;
			case('S'): key = KEY_S; break;
			case('T'): key = KEY_T; break;
			
			case('U'): key = KEY_U; break;
			case('V'): key = KEY_V; break;
			case('W'): key = KEY_W; break;
			case('X'): key = KEY_X; break;
			case('Y'): key = KEY_Y; break;
			case('Z'): key = KEY_Z; break;
			case('1'): key = KEY_1; break;
			case('2'): key = KEY_2; break;
			case('3'): key = KEY_3; break;
			case('4'): key = KEY_4; break;
			case('5'): key = KEY_5; break;
			case('6'): key = KEY_6; break;
			case('7'): key = KEY_7; break;
			case('8'): key = KEY_8; break;
			case('9'): key = KEY_9; break;
			case('0'): key = KEY_0; break;
			
			case(VK_F1): key = KEY_F1; break;
			case(VK_F2): key = KEY_F2; break;
			case(VK_F3): key = KEY_F3; break;
			case(VK_F4): key = KEY_F4; break;
			case(VK_F5): key = KEY_F5; break;
			case(VK_F6): key = KEY_F6; break;
			case(VK_F7): key = KEY_F7; break;
			case(VK_F8): key = KEY_F8; break;
			case(VK_F9): key = KEY_F9; break;
			case(VK_F10): key = KEY_F10; break;
			case(VK_F11): key = KEY_F11; break;
			case(VK_F12): key = KEY_F12; break;
			case(VK_F13): key = KEY_F13; break;
			case(VK_F14): key = KEY_F14; break;
			case(VK_F15): key = KEY_F15; break;
			case(VK_F16): key = KEY_F16; break;
			case(VK_F17): key = KEY_F17; break;
			case(VK_F18): key = KEY_F18; break;
			case(VK_F19): key = KEY_F19; break;
			case(VK_F20): key = KEY_F20; break;
			case(VK_F21): key = KEY_F21; break;
			case(VK_F22): key = KEY_F22; break;
			case(VK_F23): key = KEY_F23; break;
			case(VK_F24): key = KEY_F24; break;
			
			case('\t'): key = KEY_TAB; break;
			case(VK_CONTROL): key = KEY_CTRL; break;
			case(VK_SHIFT): key = KEY_SHIFT; break;
			
			case(VK_RETURN): key = KEY_ENTER; break;
			case(VK_SPACE): key = KEY_SPACE; break;
			case(VK_BACK): key = KEY_BACKSPACE; break;
			case(VK_DELETE): key = KEY_DEL; break;
			case(VK_UP): key = KEY_UP; break;
			case(VK_DOWN): key = KEY_DOWN; break;
			case(VK_LEFT): key = KEY_LEFT; break;
			case(VK_RIGHT): key = KEY_RIGHT; break;
			case(VK_PRIOR): key = KEY_PAGE_UP; break;
			case(VK_NEXT): key = KEY_PAGE_DOWN; break;
			case(VK_ESCAPE): key = KEY_ESCAPE; break;
			default: result = 0;
		}
		
		KeyState * state = input_state -> keys + key;
		u32 action = msg == WM_KEYDOWN ? 1 : 0;
		platform_set_key(state , action , timestamp_ms);
		
		BYTE keyboard_state[256] = {};
		
		if(GetKeyboardState(keyboard_state))
		{
			WORD ascii_buffer[2] = {};
			
			if(ToAscii(virtual_key_code , scan_code , keyboard_state , ascii_buffer , 0))
			{
				state -> character = ascii_buffer[0];
			}
		}
		
		else
			assert_zero;
		
		/*if((l_param >> 24) & 0x1)
		  {
		  key = KEY_ALT; 
		  KeyState * state = input_state -> keys + key;
		  u32 action = msg == WM_KEYDOWN ? 1 : 0;
		  platform_set_key(state , action , timestamp_ms);
		  }
		*/
	}
	
	else
	{
		result = 0;
	}
	
	return(result);
}

u32 platform_process_input_message(InputState * input_state , MSG * message , u32 screen_height)
{
	u32 result = 0;
	if(platform_process_keyboard_input(input_state , message)) result = 1;
	else if(platform_proccess_mouse_input(input_state , message , screen_height)) result = 1;
	
	return(result);
}


void platform_update_key_times(KeyState * keys , u32 size , LARGE_INTEGER timestamp)
{
	
	// NOTE: updating the state duration between frames using the last frame elapsed time.
	
	for(u32 i = 0; i < size; i++)
	{
		KeyState * state = keys + i;
		u32 index = state -> counter % KEY_DURATION_CAPACITY;
		
		//state -> timestamp.high = timestamp.HighPart;
		//state -> timestamp.low = timestamp.LowPart;
		f32 delta = platform_delta_time(timestamp);
		state -> state_durations[index] += delta;
		
		//char buffer[128] = {};
		//str_format(buffer , "delta = %\n" , delta);
		//OutputDebugString(buffer);
	}
}

void platform_reset_key_transitions(KeyState * keys , u32 size)
{
	for(u32 i = 0; i < size; i++)
	{
		keys[i].transitions = 0;
	}
	
}

void platform_preprocess_input(InputState * input_state , LARGE_INTEGER timestamp , u32 screen_x , u32 screen_y)
{
	
	
	input_state -> screen_x = screen_x;
	input_state -> screen_y = screen_y;
	
	platform_update_key_times(input_state -> keys , KEYS_INPUT_SIZE , timestamp);
	platform_reset_key_transitions(input_state -> keys , KEYS_INPUT_SIZE);
	
#if 0
	for(u32 i = 0; i < array_size(input_state -> gamepads); i++)
	{
		GamepadInput * gamepad = input_state -> gamepads + i;
		platform_update_key_times(
								  gamepad -> buttons , GAMEPAD_INPUT_SIZE , timestamp
								  );
		platform_reset_key_transitions(
									   gamepad -> buttons , GAMEPAD_INPUT_SIZE
									   );
	}
#endif
	
	input_state -> mouse.abs_last_pos_x = input_state -> mouse.abs_pos_x;
	input_state -> mouse.abs_last_pos_y = input_state -> mouse.abs_pos_y;
	
	input_state -> mouse.last_pos_x = input_state -> mouse.pos_x;
	input_state -> mouse.last_pos_y = input_state -> mouse.pos_y;
	
	input_state -> mouse.scroll_wheel = 0.0f;
}

void platform_postprocess_input(InputState * input_state)
{
	
	
	
	if(input_state -> mouse.wrapped)
	{
		input_state -> mouse.delta_x = 0;
		input_state -> mouse.delta_y = 0;
	}
	
	else
	{
		input_state -> mouse.delta_x = input_state -> mouse.abs_pos_x - input_state -> mouse.abs_last_pos_x;
		input_state -> mouse.delta_y = input_state -> mouse.abs_pos_y - input_state -> mouse.abs_last_pos_y;
	}
	
	if(input_state -> mouse.wrapping)
	{
		
		input_state -> mouse.pos_x += input_state -> mouse.delta_x;
		input_state -> mouse.pos_y += input_state -> mouse.delta_y;
	}
	
	else
	{
		input_state -> mouse.pos_x = input_state -> mouse.abs_pos_x;
		input_state -> mouse.pos_y = input_state -> mouse.abs_pos_y;
	}
	
	input_state -> mouse.wrapping = 0;
	input_state -> mouse.wrapped = 0;
}




#if 1
LRESULT platform_callback(
						  HWND handle , 
						  UINT message , 
						  WPARAM w_param , 
						  LPARAM l_param)

#else

int platform_callback()

#endif

{
	
	LRESULT result = {};
	//	assert(message != WM_SIZE);
	//	assert(message != WM_PAINT);
	
	switch(message){
		case WM_CLOSE:
		PostQuitMessage(0);
		break;
		
		case WM_DESTROY:
		PostQuitMessage(0);
		break;
		
		case WM_SIZE:
		{
			break;
		}
		
		default:
		//assert(message != WM_KEYDOWN && message != WM_KEYUP);
		result = DefWindowProc(handle , message , w_param , l_param);
	}
	
	return(result);
}

DEFINE_SET_PROCS

int
__stdcall 
WinMain(
		HINSTANCE h_instance,
		HINSTANCE h_prev_instance,
		LPSTR     lp_cmd_line,
		int       n_show_cmd)
{
	
	char dir[STR_PATH_SIZE] = {};
	PlatformProcedures procs = {};
	char * build_path = "..\\build\\";
	//char * dll_name = "main.dll";
	char * dll_path = "..\\build\\main.dll";
	char * entry_name = "entry";
	
	
	set_procs(&procs);
	set_dir("../data");
	get_dir(dir , array_size(dir));
	
	OutputDebugString(dir);
	
	char window_class_name[] = "engine_window_application";
	WNDCLASSEXA window_class = {};
	window_class.cbSize = sizeof(WNDCLASSEXA);
	window_class.style = CS_OWNDC;
	window_class.lpfnWndProc = (WNDPROC)platform_callback;
	window_class.hInstance = h_instance;
	window_class.lpszClassName = window_class_name;
	
	u32 reg_result = RegisterClassExA(&window_class);
	assert(reg_result);
	
	HWND window_handle = CreateWindowExA(
										 0 , 
										 window_class.lpszClassName , 
										 "Engine" , 
										 WS_OVERLAPPEDWINDOW , 
										 CW_USEDEFAULT , 
										 CW_USEDEFAULT , 
										 //		900 , 600  , 
										 1080 , 720  , 
										 0 , 
										 0 , 
										 h_instance , 
										 0
										 );
	
	assert(window_handle);
	UpdateWindow(window_handle);
	ShowWindow(window_handle , SW_SHOWDEFAULT);
	
	QueryPerformanceFrequency(&SYSTEM_CLOCK_FREQUENCY);
	timeBeginPeriod(1);
	HDC device_context = GetDC(window_handle);
	
	// NOTE: platform only data.
	PlatformScreenBuffer screen_buffer = {};
	BITMAPINFO screen_bitmap_info = {};
	MSG message = {};
	u32 is_running = 1;
	
	LARGE_INTEGER frame_start_time = {};
	LARGE_INTEGER last_frame_start_time = {};
	f32 frame_rate = 30;
	f32 frame_target_time = 1/(f32)frame_rate;
	f32 time_remaining = 0;
	f32 delta_time = 0;
	
	HMODULE dll_handle = {};
	void * user_data = 0;
	
	EntryFuncType * entry = load_dll_code(dll_path , "" , entry_name , &dll_handle);
	FILETIME curr_timestamp = platform_file_timestamp_by_name(dll_path , "" , FILE_LAST_WRITE_TIME);
	
	InputState input_state = {};
	
	while(is_running)
	{
		
		frame_start_time = platform_get_time();
		
		platform_resize_screen_buffer_to_window(&screen_buffer , window_handle , &screen_bitmap_info , ARGB_PIXEL_FORMAT);
		platform_preprocess_input(&input_state , last_frame_start_time , screen_buffer.width , screen_buffer.height);
		
		while(PeekMessageA(&message , 0 , 0 , 0 , PM_REMOVE))
		{
			if(message.message == WM_QUIT) is_running = 0;
			else platform_process_input_message(&input_state , &message , screen_buffer.height);
			TranslateMessage(&message);
			DispatchMessage(&message);
		}
		
		platform_postprocess_input(&input_state);
		// NOTE: doen't work unless the dll is coppied or unlocked by the debugger.
		FILETIME timestamp = platform_file_timestamp_by_name(dll_path , build_path);
		if((timestamp.dwLowDateTime != 0 && curr_timestamp.dwLowDateTime != 0) 	&&
		   (timestamp.dwLowDateTime != curr_timestamp.dwLowDateTime	|| 
		    timestamp.dwHighDateTime != curr_timestamp.dwHighDateTime))
		{
			entry = load_dll_code(dll_path , "" , entry_name , &dll_handle);
			curr_timestamp = timestamp;
		}
		
		entry(
			  0 , 0 , &is_running , 
			  screen_buffer , &input_state , delta_time , &procs , &user_data);
		
		platform_update_screen(
							   &screen_buffer , &screen_bitmap_info ,
							   device_context , screen_buffer.width , screen_buffer.height);
		
		delta_time = platform_delta_time(frame_start_time);
		time_remaining = frame_target_time - delta_time;
		
		while(time_remaining > 0)
		{
			
			Sleep((DWORD)time_remaining);
			delta_time = platform_delta_time(frame_start_time);
			time_remaining = frame_target_time - delta_time;
		}
		
		last_frame_start_time = frame_start_time;
	}
	
	return(0);
}
