/*
Author: Nathaniel Huesler
Year: 2025
Copywrite Notice: LICENSE.txt 
*/

/*

This file reads and writes logging information, such as reading from WAV files to define port data,
and producing CSV files to export internal logs. 

*/



void WAV_to_port(char * wav_filename , char * port_filename , u32 sim_time_per_second , Mem * arena)
{
	
	void * pop_addr = mem_end(*arena , void);
	Sound sound = {};
	
	load_entire_wav(&sound , arena , wav_filename);
	PlatformErrorInfo wav_file_error = get_prev_error();
	
	u32 frame_count = sound.frame_count;
	u32 stride = sound.stride_in_bytes;
	
	File port_file = open_file(port_filename , 0 , WRITE_MODE);
	PlatformErrorInfo port_file_error = get_prev_error();
	
	PortVec buffer[128] = {};
	u32 buffer_capacity = array_size(buffer);
	u32 buffer_size = 0;
	
	i32 time = 0;
	i32 sim_time_per_frame = sim_time_per_second/sound.frames_per_second;
	
	if(sound.data && port_file.platform_data)
	{
		
		if(sound.format == SOUND_PCM_I16)
		{
			
			i16 * curr_frame = (i16*)sound.data;
			i16 * end_frame = (i16*)((i8*)sound.data + frame_count*stride);
			
			while(curr_frame < end_frame)
			{	
				
				buffer_size = 0;
				while(curr_frame < end_frame && buffer_size < buffer_capacity)
				{
					
					i16 sample = *curr_frame;
					write_port_line(buffer + buffer_size++ , sample , 32 , time , 1);
					
					curr_frame = (i16*)((u8*)curr_frame + stride);
					time += sim_time_per_frame;
				}
				
				write(&port_file , buffer , buffer_size*sizeof(PortVec) , port_file.offset);
			}
		}
		
		else if(sound.format == SOUND_PCM_FP32)
		{
			
			f32 * curr_frame = (f32*)sound.data;
			f32 * end_frame = (f32*)((i8*)sound.data + frame_count*stride);
			
			while(curr_frame < end_frame)
			{
				buffer_size = 0;
				while(curr_frame < end_frame && buffer_size < buffer_capacity)
				{
					
					i16 sample = (*curr_frame)*MAX_I16;
					write_port_line(buffer + buffer_size++ , sample , 32 , time , 1);
					
					curr_frame = (f32*)((u8*)curr_frame + stride);
					time += sim_time_per_frame;
				}
				
				write(&port_file , buffer , buffer_size*sizeof(PortVec) , port_file.offset);
			}
		}
		
		else
		{
			report_error("unsupported sound format in '%'" , 1 , wav_filename);
		}
	}
	
	if(!sound.data)
	{
		report_error("cound not open sound file '%' : %" , 1 , wav_filename , wav_file_error.message);
	}
	
	if(!port_file.platform_data)
	{
		report_error("cound not open port file '%' : %" , 1 , port_filename , port_file_error.message);
	}
	
	close_file(&port_file);
	mem_pop(arena , pop_addr);
}


void port_to_WAV(
				 char * wav_filename , char * port_filename , u32 port_width , u32 sim_time_per_second , Mem * arena)
{
	
	File wav_file = open_file(wav_filename , 0 , WRITE_MODE); 
	PlatformErrorInfo wav_file_error = get_prev_error();
	
	File port_file = open_file(port_filename , 0 , READ_MODE);
	PlatformErrorInfo port_file_error = get_prev_error();
	
	if(wav_file.platform_data && port_file.platform_data)
	{
		
		u32 file_size = get_file_size_by_handle(port_file.platform_data);
		u32 vec_count = file_size / sizeof(PortVec);
		PortVec * vecs = mem_push_array(arena , PortVec , vec_count);
		read(&port_file , vecs , vec_count*sizeof(PortVec) , 0);
		
		Sound sound = create_empty_sound(SOUND_PCM_I16 , 44100 , 1 , vec_count , arena);
		u32 vec_index = 0;
		i16 * samples = (i16*)sound.data;
		WAVFileInfo info = {};
		
		while(vec_index < vec_count)
		{
			PortVec vec = vecs[vec_index];
			CircuitValue cv = {vec.circuit_value , port_width};
			i16 sample = cv_to_int(cv , 1);
			
			samples[vec_index++] = sample;
		}
		
		if(!save_wav(&info , &sound , &wav_file  , 0))
		{
			report_error("could not save sond samples to %" , 1 , wav_filename);
		}
	}
	
	if(!wav_file.platform_data)
	{
		report_error("cound not open sound file '%' : %" , 1 , wav_filename , wav_file_error.message);
	}
	
	if(!port_file.platform_data)
	{
		report_error("cound not open port file '%' : %" , 1 , port_filename , port_file_error.message);
	}
}


inline
File start_csv_file(char * filename , char * header)
{
	
	File file = open_file(filename , 0 , WRITE_MODE);
	write(&file , header , str_size(header) , 0);
	write(&file , "\n" , 1 , file.offset);
	
	return(file);
}

inline
void end_csv_file(File * file)
{
	close_file(file);
}


void update_frame_power_csv(File * file , i32 * last_frame_time_ptr , StatFrame * frames , u32 frame_count)
{
	
	i32 last_frame_time = *last_frame_time_ptr;
	i32 next_frame_time = *last_frame_time_ptr;
	
	char buffer[256] = {};
	
	for(u32 frame_index = 0; frame_index < frame_count; frame_index++)
	{
		
		StatFrame * frame = frames + frame_index;
		i32 time = frame -> time;
		u32 static_energy = frame -> norm_static_energy;
		u32 switch_energy = frame -> norm_switch_energy;
		u32 short_circuit_energy = frame -> norm_short_circuit_energy;
		u32 switches = frame -> switches;
		u32 fanout = frame -> fanout;
		char * curr_char = buffer;
		
		if(time > last_frame_time)
		{
			curr_char += str_format(
									curr_char , "% , % , % , % , % , %\n" , 
									time , static_energy , switch_energy , short_circuit_energy , switches , fanout
									);
			
			write(file , buffer , curr_char-buffer , file -> offset);
		}
		
		if(next_frame_time < time) next_frame_time = time;
	}
	
	*last_frame_time_ptr = next_frame_time;
}


void update_module_power_csv(File * file , i32 * last_module_time_ptr , Mem power_arena)
{
	
	i32 last_module_time = *last_module_time_ptr;
	i32 next_module_time = *last_module_time_ptr;
	
	CircuitModulePower * modules = mem_start(power_arena , CircuitModulePower);
	u32 module_count = mem_size(power_arena , CircuitModulePower);
	char buffer[256] = {};
	
	for(u32 module_index = 0; module_index < module_count; module_index++)
	{
		
		CircuitModulePower * module = modules + module_index;
		i32 time = module -> time;
		u32 id = module -> module_id;
		u32 static_energy = module -> norm_static_energy;
		u32 switch_energy = module -> norm_switch_energy;
		u32 switches = module -> switches;
		u32 fanout = module -> fanout;
		char * curr_char = buffer;
		
		if(time > last_module_time)
		{
			curr_char += str_format(
									curr_char , "% , % , % , % , % , %\n" , 
									time , id , static_energy , switch_energy , switches , fanout
									);
			
			write(file , buffer , curr_char-buffer , file -> offset);
		}
		
		if(next_module_time < time) next_module_time = time;
	}
	
	*last_module_time_ptr = next_module_time;
}



void update_addr_power_csv(File * file , i32 * last_addr_time_ptr , Mem power_arena)
{
	
	i32 last_addr_time = *last_addr_time_ptr;
	i32 next_addr_time = *last_addr_time_ptr;
	
	CircuitAddressPower * power_list = mem_start(power_arena , CircuitAddressPower);
	u32 power_list_count = mem_size(power_arena , CircuitAddressPower);
	char buffer[256] = {};
	
	for(u32 power_index = 0; power_index < power_list_count; power_index++)
	{
		
		CircuitAddressPower * power = power_list + power_index;
		i32 time = power -> time;
		CircuitAddress addr = power -> addr;
		u32 switches = power -> switches;
		u32 fanout = power -> fanout;
		char * curr_char = buffer;
		
		if(time > last_addr_time)
		{
			if(switches)
			{
				
				curr_char += str_format(
										curr_char , 
										"% , % , % , "
										"% , %\n" , 
										time , addr.min , addr.max , switches , fanout
										);
				
				write(file , buffer , curr_char-buffer , file -> offset);
			}
		}
		
		if(next_addr_time < time) next_addr_time = time;
	}
	
	*last_addr_time_ptr = next_addr_time;
}


void update_stage_csv(File * file , i32 * last_stage_time_ptr , InstrTracker * tracker , EventBuffer * event_buffer)
{
	
	i32 last_stage_time = *last_stage_time_ptr;
	i32 next_stage_time = *last_stage_time_ptr;
	
	InstrStage * stage_list = tracker -> stage_list;
	u32 stage_count = tracker -> stage_count;
	
	CircuitEvent * events = event_buffer -> events;
	char buffer[256] = {};
	
	for(u32 stage_index = 0; stage_index < stage_count; stage_index++)
	{
		InstrStage * stage = stage_list + stage_index;
		u32 PC = stage -> pc_value;
		u32 id = stage -> id;
		
		CircuitEvent * start_event = events + stage -> start_event_index;
		CircuitEvent * ss_event = events + stage -> ss_event_index;
		CircuitEvent * end_event = events + stage -> end_event_index;
		
		i32 start_time = start_event -> time;
		i32 ss_time = ss_event -> time;
		i32 end_time = end_event -> time;
		
		char * curr_char = buffer;
		
		if(start_time > last_stage_time)
		{
			curr_char += str_format(
									curr_char , "% , % , % , % , %\n" , 
									start_time , ss_time , end_time , id , PC
									);
			
			write(file , buffer , curr_char-buffer , file -> offset);
		}
		
		if(next_stage_time < start_time) next_stage_time = start_time;
		
	}
	
	*last_stage_time_ptr = next_stage_time;
}



void save_stats(
				StatTracker * tracker , CircuitPort * ports , u32 port_count , i32 time , 
				char * filename , Mem * arena)
{
	
	File file = open_file(filename , 0 , WRITE_MODE);
	char * buffer = mem_end(*arena , char);
	char * curr_char = buffer;
	
	u32 frames_recorded = tracker -> frames_recorded;
	u32 stages_recorded = tracker -> stages_recorded;
	u32 instrs_recorded = tracker -> instrs_recorded;
	u32 total_energy = tracker -> total_energy;
	u32 total_stage_period = tracker -> total_stage_period;
	
	u32 max_energy = tracker -> max_energy;
	u32 max_stage_period = tracker -> max_stage_period;
	
	u32 avg_stage_period = 0;
	u32 avg_energy = 0;
	
	if(frames_recorded) avg_energy = total_energy / frames_recorded;
	if(stages_recorded) avg_stage_period = total_stage_period / stages_recorded;
	
	curr_char += str_format(
							curr_char , 
							"total sim-cycles = %{del:3}\n"
							"total frames = %{del:3}\n"
							"total stages = %{del:3}\n"
							"total instructions = %{del:3}\n"
							"max energy = %\n"
							"avg energy = %\n"
							"max stage period = %\n"
							"avg stage period = %\n" , 
							time , frames_recorded , stages_recorded , instrs_recorded , max_energy , avg_energy , 
							max_stage_period , avg_stage_period
							);
	
	for(u32 i = 0; i < port_count; i++)
	{
		
		CircuitPort * port = ports + i;
		u32 total_time = port -> total_time;
		u32 trigger_count = port -> trigger_count;
		f32 sample_period = 0;
		
		if(trigger_count) sample_period = total_time / trigger_count;
		
		curr_char += str_format(
								curr_char , 
								"port % , samples = %{del:3} , sim-cycles = %{del:3} , sample-period = %\n" , 
								i , trigger_count , total_time , sample_period
								);
	}
	
	write(&file , buffer , curr_char-buffer , 0);
	close_file(&file);
	
	
}

