/*
Author: Nathaniel Huesler
Year: 2025
Copywrite Notice: LICENSE.txt 
*/

/*

This file encodes the MIPS-ISA and generates the processor. It manages modules such as 
the register file, the ALU unit and the control logic. It defines the structure wherein 
optional modules may be embedded such as the speacial DOT product unit. It defines a simple
pipeline, with memory and the clock. 

*/


#define CIRCUIT_NULL_ADDR CircuitAddress{}
#define GND CircuitAddress{8 , 8+32}
#define VDD CircuitAddress{8+32 , 8+64}

#define MAX_LINE_ENUM 999999


enum
{
	MIPS_PATH = 1 , 
	DOT4_PATH
};

// NOTE: line id's

enum
{
	
	
	CLOCK_LINE = 1 , 
	
	// stage lines
	FETCH_STAGE_LINE , 
	DECODE_STAGE_LINE , 
	EXECUTE_STAGE_LINE , 
	
	// control lines
	PC_LINE , 
	IR_LINE , 
	IR_OP_CODE_LINE , 
	IR_RD_LINE , 
	IR_RT_LINE , 
	IR_RS_LINE , 
	IR_IMM_LINE_DIRECT , 
	IR_JUMP_LINE_DIRECT , 
	
	IR_IMM_LINE ,  // NOTE; these are sign extended 
	IR_JUMP_LINE , 
	
	REG_WRITE_ENABLE_LINE , 
	REG_WRITE_ADDR_LINE , 
	REG_WRITE_DATA_LINE , 
	
	PC_WRITE_ENABLE_LINE , 
	BRANCH_LINE , 
	JUMP_LINE , 
	
	// misc
	PORT0_LINE , 
	PORT1_LINE , 
	
	// MIPS path
	MIPS_PATH_ENABLE_LINE , 
	MIPS_REG_WRITE_ENABLE_LINE , 
	MIPS_REG_WRITE_ADDR_LINE , 
	MIPS_REG_WRITE_DATA_LINE , 
	
	MIPS_PC_WRITE_ENABLE_LINE , 
	MIPS_BRANCH_LINE , 
	MIPS_JUMP_LINE , 
	
	MIPS_ALU_FUNC_LINE , 
	MIPS_WB_SEL_LINE , 
	MIPS_PC_SEL_LINE , 
	MIPS_DATA_MEM_READ_LINE , 
	MIPS_DATA_MEM_WRITE_LINE , 
	MIPS_IMM_SEL_LINE , 
	MIPS_REG_WRITE_SEL_LINE , 
	
	MIPS_REG_OUTPUT_DATA_1_LINE , 
	MIPS_REG_OUTPUT_ADDR_1_LINE , 
	
	MIPS_REG_INPUT_DATA_1_LINE , 
	MIPS_REG_INPUT_ADDR_1_LINE , 
	
	MIPS_REG_INPUT_DATA_2_LINE , 
	MIPS_REG_INPUT_ADDR_2_LINE , 
	
	MIPS_ALU_OUT_LINE , 
	MIPS_MULT_OUT_LINE , 
	MIPS_ADDER_OUT_LINE , 
	MIPS_SHIFT_LEFT_OUT_LINE , 
	MIPS_SHIFT_RIGHT_OUT_LINE , 
	
	// DOT4 path
	DOT4_PATH_ENABLE_LINE , 
	DOT4_REG_WRITE_ENABLE_LINE , 
	
};

enum
{
	INSTR_STAGE_UNKNOWN , 
	INSTR_STAGE_FETCH , 
	INSTR_STAGE_DECODE , 
	INSTR_STAGE_EXECUTE , 
};

enum
{
	INSTR_NO_TYPE , 
	INSTR_R_TYPE , 
	INSTR_I_TYPE , 
	INSTR_J_TYPE
};

enum
{
	FIELD_REGISTER = 1 , 
	FIELD_CONSTANT
};

enum
{
	OP_HALT , 
	OP_ADD_I , 
	OP_ADD , 
	OP_MULT , 
	OP_SHIFT_LEFT , 
	OP_SHIFT_RIGHT , 
	OP_JUMP , 
	OP_LOAD , 
	OP_STORE , 
	
	OP_DOT4 , 
	
	OP_COUNT , 
};

enum
{
	ALU_ADD , 
	ALU_MULT , 
	ALU_SHIFT_LEFT , 
	ALU_SHIFT_RIGHT , 
};

enum
{
	WB_SEL_ALU , 
	WB_SEL_DATA_MEM
};


enum
{
	PC_INCR , 
	PC_BRANCH , 
	PC_STOP
};


enum
{
	ZR_REGISTER , 
	GP_REGISTER , 
	IO_REGISTER
};


enum
{
	UNSIGNED_VALUE = 0 , 
	SIGNED_VALUE = 1
};


struct Instr
{
	
	u32 op_code;
	u32 rd;
	u32 rs;
	u32 rt;
	i32 imm;
	i32 pc_rel_jump;
	
	CircuitValue encoded_cv;
};


struct LineSpec
{
	
	u32 id;
	u32 width;
	char name[64];
	CircuitAddress addr;
};

struct StageLineSpec
{
	u32 line_id;
	u32 stage_id;
	char name[64];
	u32 width;
};

struct RegisterSpec
{
	u32 type;
	u32 select_addr;
	u32 port_id;
	u32 line_id;
	char name[64];
};

struct PortSpec 
{
	u32 direction;
	char filename[STR_PATH_SIZE];
	char name[64];
	
	u32 line_id;
	u32 width;
	u32 sign;
	u32 min;
	u32 max;
};

struct InstrFieldSpec 
{
	
	u32 type;
	u32 line_id;
	char name[64];
	u32 min;
	u32 max;
	u32 sign;
};

struct InstrSpec 
{
	char mnemonic[64];
	u32 op_code;
	u32 type;
	
	u32 * ctr_line_ids;
	u32 * ctr_line_values;
	u32 ctr_line_count;
	u32 ctr_line_capacity;
};




struct ControlUnitSpec
{
	
	u32 * ctr_lines;
	u32 ctr_line_count;
	u32 ctr_line_capacity;
	
	// NOTE: generated data
	CircuitAddress * ctr_bit_addrs;
	u32 * op_codes;
	u32 * ctr_truth_table;
	
	u32 ctr_bit_count;
	u32 op_code_count;
};

struct DataPathSpec
{
	
	u32 id;
	
	//LineSpecSet line_set;
	//InstrSpecSet instr_set;
	ControlUnitSpec ctr_unit;
	
	u32 hw_instr_start;
	u32 hw_instr_end;
	
	u32 sim_mem_start;
	u32 sim_mem_end;
	
	u32 instr_spec_start;
	u32 instr_spec_end;
	
	u32 * instr_indices;
	u32 instr_capacity;
	u32 instr_count;
	
	u32 enable_line;
	
	
};

struct MipsSpec
{
	
	Mem lines;
	Mem regs;
	Mem stage_lines;
	Mem ports;
	Mem fields;
	Mem instr_set;
	Mem paths;
	
	Mem instr_spec_ctr_line_arena;
	
	i32 max_imm_value;
	i32 min_jump_value;
	i32 max_jump_value;
	u32 reg_width;
	u32 reg_addr_width;
	u32 instr_width;
};



struct CircuitModule
{
	char name[STR_NAME_SIZE];
	u32 id;
	
	u32 min_instr_index;
	u32 max_instr_index;
	
	CircuitAddress * addr_list;
	u32 addr_count;
	
};

struct CircuitModuleLink
{
	u32 parent_index;
	u32 child_index;
};

struct CircuitProgram
{
	
	HardwareInstr * hw_instrs;
	u32 hw_instr_count;
	u32 hw_instr_capacity;
	
	CircuitMemoryInfo * circuit_mem_info;
	u32 circuit_mem_info_count;
	u32 circuit_mem_info_capacity;
	
	CircuitPort * ports;
	u32 port_count;
	u32 port_capacity;
	
	CircuitInputVector * input_vecs;
	u32 input_vec_count;
	u32 input_vec_capacity;
	
	CircuitModule * modules;
	u32 module_count;
	u32 module_capacity;
	
	CircuitModuleLink * module_links;
	u32 module_link_count;
	u32 module_link_capacity;
	
	Mem module_addr_list_arena;
	
	u32 * module_stack;
	u32 module_stack_size;
	u32 module_stack_capacity;
	
	TestVector * test_vecs;
	u32 test_vec_count;
	u32 test_vec_capacity;
	
	TestVectorExpr * test_vec_exprs;
	u32 test_vec_expr_count;
	u32 test_vec_expr_capacity;
	
	SignalLog * test_vec_buffer_mem;
	u32 test_vec_buffer_mem_count;
	u32 test_vec_buffer_mem_capacity;
	
	SignalTrace * traces;
	u32 trace_count;
	u32 trace_capacity;
	
	SignalTraceResult * trace_results;
	u32 trace_result_count;
	u32 trace_result_capacity;
	
	u8 * path_mem;
	u32 path_mem_size;
	u32 path_mem_capacity;
	
	u32 curr_byte_addr;
	i32 curr_time;
	
};


struct CircuitWatchPoint
{
	CircuitAddress addr;
	CircuitValue value;
	char * name;
};


CircuitInputVector * push_input_vector(
									   CircuitProgram * prog , CircuitAddress addr , CircuitValue value)
{
	
	assert(width_of(value) == width_of(addr));
	CircuitInputVector * vecs = prog -> input_vecs;
	u32 count = prog -> input_vec_count;
	u32 capacity = prog -> input_vec_capacity;
	
	assert(count < capacity);
	
	CircuitInputVector * input = vecs + count++;
	
	input -> addr = addr;
	input -> value = value;
	input -> time = prog -> curr_time;
	
	prog -> input_vec_count = count;
	
	return(input);
}

CircuitInputVector * push_input_vector(
									   CircuitProgram * prog , char * name , CircuitAddress sub_addr , 
									   CircuitValue value)
{
	
	CircuitMemoryInfo * mem_info = prog -> circuit_mem_info;
	u32 mem_info_count = prog -> circuit_mem_info_count;
	
	CircuitMemoryInfo * info = get_circuit_mem_info(mem_info , mem_info_count , name);
	assert(info);
	
	sub_addr.min += info -> addr.min;
	sub_addr.max += info -> addr.min;
	assert(in_range(sub_addr , info -> addr));
	
	CircuitInputVector * input = push_input_vector(prog , sub_addr , value);
	
	return(input);
}

CircuitInputVector * push_input_vector(
									   CircuitProgram * prog , CircuitAddress addr , u32 value)
{
	CircuitInputVector * input = push_input_vector(
												   prog , addr , {value , width_of(addr)}
												   );
	
	return(input);
}

CircuitInputVector *  push_input_vector(
										CircuitProgram * prog , char * name , CircuitAddress sub_addr , u32 value)
{
	CircuitInputVector * input = push_input_vector(
												   prog , name , sub_addr , CircuitValue{value , width_of(sub_addr)}
												   );
	
	return(input);
}

inline
void push_time(CircuitProgram * prog)
{
	prog -> curr_time++;
}


void add_moulde_addesses(
						 CircuitAddress * add_addrs , u32 add_count , 
						 CircuitAddress * addr_list , u32 * addr_count_ptr , u32 addr_capacity)
{
	
	u32 addr_count = *addr_count_ptr;
	for(u32 add_index = 0; add_index < add_count; add_index++)
	{
		
		u32 found = 0;
		
		for(u32 addr_index = 0; addr_index < addr_count; addr_index++)
		{
			if(add_addrs[add_index] == addr_list[addr_index])
			{
				found = 1;
				break;
			}
		}
		
		if(!found)
		{
			assert(addr_count < addr_capacity);
			addr_list[addr_count++] = add_addrs[add_index];
		}
	}
	
	*addr_count_ptr = addr_count;
}


void get_module_addresses(CircuitModule * module , HardwareInstr * instrs , u32 instr_count , Mem * arena)
{
	
	CircuitAddress * addr_list = mem_end(*arena , CircuitAddress);
	u32 capacity = mem_rem(*arena , CircuitAddress);
	u32 count = 0;
	
	u32 instr_min = module -> min_instr_index;
	u32 instr_max = module -> max_instr_index;
	
	for(u32 instr_index = instr_min; instr_index < instr_max; instr_index++)
	{
		
		HardwareInstr * instr = instrs + instr_index;
		HardwareInstrInfo * info = get_hw_instr_info(instr -> function);
		u32 input_count = info -> input_count;
		u32 output_count = info -> output_count;
		
		add_moulde_addesses(instr -> inputs , input_count , addr_list , &count , capacity);
		add_moulde_addesses(instr -> outputs , output_count , addr_list , &count , capacity);
	}
	
	module -> addr_list = addr_list;
	module -> addr_count = count;
	mem_push(arena , CircuitAddress , sizeof(CircuitAddress)*count);
}



inline 
void push_module(CircuitProgram * prog , char * name)
{
	
	u32 module_count = prog -> module_count;
	u32 module_capacity = prog -> module_capacity;
	
	u32 link_count = prog -> module_link_count;
	u32 link_capacity = prog -> module_link_capacity;
	
	u32 stack_size = prog -> module_stack_size;
	u32 stack_capacity = prog -> module_stack_capacity;
	
	u32 * stack = prog -> module_stack;
	CircuitModule * modules = prog -> modules;
	CircuitModuleLink * links = prog -> module_links;
	
	assert(module_count < module_capacity);
	assert(link_count < link_capacity);
	assert(stack_size < stack_capacity);
	
	CircuitModule * module = modules + module_count;
	
	str_copy(module -> name , name);
	module -> min_instr_index = prog -> hw_instr_count;
	module -> id = module_count+1;
	
	if(stack_size)
	{
		CircuitModuleLink * link = links + link_count++;
		link -> parent_index = stack[stack_size-1];
		link -> child_index = module_count;
	}
	
	stack[stack_size++] = module_count++;
	
	prog -> module_count = module_count;
	prog -> module_link_count = link_count;
	prog -> module_stack_size = stack_size;
}


inline 
void pop_module(CircuitProgram * prog)
{
	
	u32 stack_size = prog -> module_stack_size;
	u32 * stack = prog -> module_stack;
	CircuitModule * modules = prog -> modules;
	u32 module_count = prog -> module_count;
	
	assert(stack_size);
	u32 index = stack[stack_size-1];
	assert(index < module_count);
	
	CircuitModule * module = modules + index;
	module -> max_instr_index = prog -> hw_instr_count;
	
	get_module_addresses(module , prog -> hw_instrs , prog -> hw_instr_count , &prog -> module_addr_list_arena);
	
	stack[--stack_size] = 0;
	
	prog -> module_stack_size = stack_size;
}



inline
void name_addr(CircuitProgram * prog , char * name , CircuitAddress addr , u32 polarity)
{
	assert(prog -> circuit_mem_info_count < prog -> circuit_mem_info_capacity);
	CircuitMemoryInfo * info = prog -> circuit_mem_info + prog -> circuit_mem_info_count++;
	
	str_copy(info -> name , name);
	info -> addr = addr;
	info -> polarity = polarity;
	
}


inline
CircuitAddress push_addr(CircuitProgram * prog , char * name , u32 bit_width , u32 polarity)
{
	
	CircuitAddress addr = {};
	u32 curr_byte_addr = prog -> curr_byte_addr;
	u32 byte_width = get_byte_width(bit_width);
	
	addr.min = curr_byte_addr*8;
	addr.max = curr_byte_addr*8 + bit_width;
	curr_byte_addr += byte_width;
	
	name_addr(prog , name , addr , polarity);
	
	prog -> curr_byte_addr = curr_byte_addr;
	
	return(addr);
}

inline
CircuitAddress push_addr(CircuitProgram * prog , u32 bit_width)
{
	CircuitAddress addr = {};
	u32 curr_byte_addr = prog -> curr_byte_addr;
	u32 byte_width = get_byte_width(bit_width);
	
	addr.min = curr_byte_addr*8;
	addr.max = curr_byte_addr*8 + bit_width;
	curr_byte_addr += byte_width;
	
	prog -> curr_byte_addr = curr_byte_addr;
	
	return(addr);
}


inline 
HardwareInstr * push_hw_instruction(
									CircuitProgram * prog , u32 function , u32 input_count , u32 output_count)
{
	assert(prog -> hw_instr_count < prog -> hw_instr_capacity);
	assert(function < COMP_COUNT);
	
	HardwareInstr * instr = prog -> hw_instrs + prog -> hw_instr_count++;
	instr -> function = function;
	
	HardwareInstrInfo * info = get_hw_instr_info(function);
	assert(info);
	assert(info -> input_count == input_count);
	assert(info -> output_count == output_count);
	
	return(instr);
}

TestVector * push_test_vec(CircuitProgram * prog , u32 conseq_count , u32 antec_count)
{
	
	assert(prog -> test_vec_count < prog -> test_vec_capacity);
	
	assert(conseq_count < TEST_VECTOR_CONSEQ_CAPACITY);
	assert(antec_count < TEST_VECTOR_ANTEC_CAPACITY);
	
	TestVector * vec = prog -> test_vecs + prog -> test_vec_count;
	vec -> exprs = prog -> test_vec_exprs + prog -> test_vec_expr_count;
	vec -> expr_count = 0;
	
	vec -> conseq_count = 0;
	vec -> antec_count = 0;
	
	prog -> test_vec_count++;
	
	return(vec);
}


TestVectorExpr * push_expr(CircuitProgram * prog)
{
	u32 expr_count = prog -> test_vec_expr_count;
	u32 expr_capacity = prog -> test_vec_expr_capacity;
	TestVectorExpr * exprs = prog -> test_vec_exprs;
	
	assert(expr_count < expr_capacity);
	
	TestVectorExpr * expr = exprs + expr_count++;
	prog -> test_vec_expr_count = expr_count;
	
	return(expr);
}

TestVectorExpr * make_expr(TestVectorExpr * lhs , TestVectorExpr * rhs , u32 op_code , CircuitProgram * prog)
{
	
	//assert(width_of(lhs) == width_of(rhs));
	
	TestVectorExpr * expr = push_expr(prog);
	
	expr -> lhs = lhs;
	expr -> rhs = rhs;
	expr -> op_code = op_code;
	
	return(expr);
}


TestVectorExpr * make_expr(CircuitAddress addr , u32 op_code , TimeSpec time_spec , CircuitProgram * prog)
{
	
	assert(width_of(addr) <= 32);
	
	TestVectorExpr * expr = push_expr(prog);
	expr -> addr = addr;
	expr -> op_code = op_code;
	expr -> time_spec = time_spec;
	
	return(expr);
}

TestVectorExpr * make_expr(CircuitValue value , u32 op_code , TimeSpec time_spec , CircuitProgram * prog)
{
	
	assert(width_of(value) <= 32);
	
	TestVectorExpr * expr = push_expr(prog);
	expr -> value = value;
	expr -> op_code = op_code;
	expr -> time_spec = time_spec;
	
	return(expr);
}


TestVectorExpr * make_expr(
						   CircuitAddress lhs_addr , TestVectorExpr * rhs , 
						   u32 op_code , TimeSpec lhs_time_spec ,
						   CircuitProgram * prog)
{
	
	TestVectorExpr * lhs = make_expr(lhs_addr , EXPR_OPERAND , lhs_time_spec , prog);
	TestVectorExpr * result = make_expr(lhs , rhs , op_code , prog);
	
	return(result);
}


TestVectorExpr * make_expr(
						   CircuitAddress lhs_addr , CircuitAddress rhs_addr , 
						   u32 op_code , TimeSpec lhs_time_spec , TimeSpec rhs_time_spec , 
						   CircuitProgram * prog)
{
	
	TestVectorExpr * lhs = make_expr(lhs_addr , EXPR_OPERAND , lhs_time_spec , prog);
	TestVectorExpr * rhs = make_expr(rhs_addr , EXPR_OPERAND , rhs_time_spec , prog);
	TestVectorExpr * result = make_expr(lhs , rhs , op_code , prog);
	
	return(result);
}


TestVectorExpr * make_expr(
						   CircuitAddress lhs_addr , CircuitValue rhs_value , 
						   u32 op_code , TimeSpec lhs_time_spec , TimeSpec rhs_time_spec , 
						   CircuitProgram * prog)
{
	
	TestVectorExpr * lhs = make_expr(lhs_addr , EXPR_OPERAND , lhs_time_spec , prog);
	TestVectorExpr * rhs = make_expr(rhs_value , EXPR_OPERAND , rhs_time_spec , prog);
	TestVectorExpr * result = make_expr(lhs , rhs , op_code , prog);
	
	return(result);
}

TestVectorExpr * make_expr(
						   CircuitAddress lhs_addr , u32 rhs_value_ , 
						   u32 op_code , TimeSpec lhs_time_spec , TimeSpec rhs_time_spec , 
						   CircuitProgram * prog)
{
	
	
	CircuitValue rhs_value = {rhs_value_ , width_of(lhs_addr)};
	TestVectorExpr * lhs = make_expr(lhs_addr , EXPR_OPERAND , lhs_time_spec , prog);
	TestVectorExpr * rhs = make_expr(rhs_value , EXPR_OPERAND , rhs_time_spec , prog);
	TestVectorExpr * result = make_expr(lhs , rhs , op_code , prog);
	
	return(result);
}



void push_conseq(TestVectorExpr * expr , TestVector * vec , CircuitProgram * prog)
{
	assert(vec -> conseq_count < TEST_VECTOR_CONSEQ_CAPACITY);
	vec -> conseq[vec -> conseq_count++] = expr;
	
	vec -> expr_count = prog -> test_vec_exprs + prog -> test_vec_expr_count - vec -> exprs;
}

void push_antec(TestVectorExpr * expr , TestVector * vec , CircuitProgram * prog)
{
	assert(vec -> antec_count < TEST_VECTOR_CONSEQ_CAPACITY);
	vec -> antec[vec -> antec_count++] = expr;
	
	vec -> expr_count = prog -> test_vec_exprs + prog -> test_vec_expr_count - vec -> exprs;
}



struct TraceBounds
{
	TimeSpec start;
	TimeSpec end;
};

u32 push_trace_result(CircuitProgram * prog)
{
	
	u32 count = prog -> trace_result_count;
	u32 capacity = prog -> trace_result_capacity;
	assert(capacity <= MAX_TRACE_ID); 
	
	u32 index = MIN_TRACE_ID + count++;
	
	assert(count < capacity);
	prog -> trace_result_count = count;
	
	return(index);
}

CircuitAddress * push_path(CircuitAddress * path_in , u32 path_length , CircuitProgram * prog)
{
	
	u8 * path_mem = prog -> path_mem;
	u32 size = prog -> path_mem_size;
	u32 capacity = prog -> path_mem_capacity;
	
	
	u32 bytes = path_length*sizeof(CircuitAddress);
	CircuitAddress * path = (CircuitAddress*)(path_mem + size);
	
	assert(size + bytes < capacity);
	mem_copy(path , path_in , bytes);
	size += bytes;
	
	prog -> path_mem_size = size;
	
	return(path);
}

TraceBounds make_backtrace(CircuitAddress * path , u32 path_length , CircuitProgram * prog)
{
	
	TraceBounds bounds = {};
	
	u32 count = prog -> trace_count;
	u32 capacity = prog -> trace_capacity;
	SignalTrace * traces = prog -> traces;
	
	u32 start_id = push_trace_result(prog);
	u32 end_id = push_trace_result(prog);
	
	assert(count < capacity);
	SignalTrace * trace = traces + count++;
	
	trace -> path = push_path(path , path_length , prog);
	trace -> path_length = path_length;
	trace -> start_result_id = start_id;
	trace -> end_result_id = end_id;
	
	bounds.start.id = start_id;
	bounds.end.id = end_id;
	prog -> trace_count = count;
	
	return(bounds);
}





void check_integrity(CircuitProgram * prog , MipsSpec * spec)
{
	
	assert(prog -> module_stack_size == 0);
	
	TestVector * tests = prog -> test_vecs;
	u32 test_count = prog -> test_vec_count;
	
	HardwareInstr * instrs = prog -> hw_instrs;
	u32 instr_count = prog -> hw_instr_count;
	
	// NOTE: check that the categorised expressions (conseq,antec) are within the total expressions array.
	for(u32 i = 0; i < test_count; i++)
	{
		
		TestVector * target = tests + i;
		TestVectorExpr * min_expr = target -> exprs;
		TestVectorExpr * max_expr = target -> exprs + target -> expr_count;
		
		TestVectorExpr ** conseq = target -> conseq;
		TestVectorExpr ** antec = target -> antec;
		
		u32 conseq_count = target -> conseq_count;
		u32 antec_count = target -> antec_count;
		
		assert(min_expr <= max_expr);
		
		for(u32 conseq_index = 0; conseq_index < conseq_count; conseq_index++)
		{
			TestVectorExpr * expr = conseq[conseq_index];
			assert(min_expr <= expr && expr < max_expr);
		}
		
		for(u32 antec_index = 0; antec_index < antec_count; antec_index++)
		{
			TestVectorExpr * expr = antec[antec_index];
			assert(min_expr <= expr && expr < max_expr);
		}
	}
	
	// NOTE: check that the categorised expressions (conseq,antec) are in order in total expressions array.
	for(u32 i = 0; i < test_count; i++)
	{
		
		for(u32 j = i+1; j < test_count; j++)
		{
			
			TestVector * target = tests + i;
			TestVector * curr = tests + j;
			
			TestVectorExpr * t_min = target -> exprs;
			TestVectorExpr * t_max = target -> exprs + target -> expr_count;
			
			TestVectorExpr * c_min = curr -> exprs;
			//TestVectorExpr * c_max = curr -> exprs + curr -> expr_count;
			
			assert(t_min < c_min && t_max <= c_min);
		}
	}
	
	
	// NOTE: check that there are no connections output to VDD and GND.
	
	for(u32 i = 0; i < instr_count; i++)
	{
		HardwareInstr * curr_instr = instrs + i;
		HardwareInstrInfo * info = get_hw_instr_info(curr_instr -> function);
		
		for(u32 output_index = 0; output_index < info -> output_count; output_index++)
		{
			assert(!in_range(curr_instr -> outputs[output_index] , VDD));
			assert(!in_range(curr_instr -> outputs[output_index] , GND));
		}
	}
	
	
	// NOTE: check that all lines have been allocated
	LineSpec * lines = mem_start(spec -> lines , LineSpec);
	u32 line_count = mem_size(spec -> lines , LineSpec);
	
	for(u32 line_index = 0; line_index < line_count; line_index++)
	{
		
		LineSpec * curr_line = lines + line_index;
		CircuitAddress addr = curr_line -> addr;
		assert(addr.min || addr.max);
	}
	
}

LineSpec * push_line_spec(Mem * arena , u32 id , char * name , u32 width)
{
	
	LineSpec * lines = mem_start(*arena , LineSpec);
	u32 count = mem_size(*arena , LineSpec);
	
	for(u32 line_index = 0; line_index < count; line_index++)
	{
		assert(lines[line_index].id != id);
		assert(!str_eql(lines[line_index].name , name));
	}
	
	
	LineSpec * line = mem_push_struct(arena , LineSpec);
	*line = {};
	
	line -> id = id;
	line -> width = width;
	str_copy(line -> name , name);
	
	return(line);
	
}

LineSpec * get_line_spec(Mem line_arena , u32 id)
{
	LineSpec * result = {};
	LineSpec * lines = mem_start(line_arena , LineSpec);
	u32 line_count = mem_size(line_arena , LineSpec);
	
	for(u32 line_index = 0; line_index < line_count; line_index++)
	{
		LineSpec * curr_line = lines + line_index;
		if(curr_line -> id == id)
		{
			result = curr_line;
		}
	}
	
	return(result);
}

CircuitAddress get_line_addr(Mem line_arena , u32 id)
{
	
	LineSpec * line = get_line_spec(line_arena , id);
	CircuitAddress addr = line -> addr;
	assert(addr.min | addr.max);
	
	return(addr);
}


CircuitAddress push_line_addr(CircuitProgram * prog , Mem line_arena , u32 id , u32 width)
{
	
	LineSpec * line = get_line_spec(line_arena , id);
	assert(!line -> addr.min && !line -> addr.max);
	
	line -> addr = push_addr(prog , line -> name , width , 0);
	return(line -> addr);
}


void set_line_addr(CircuitProgram * prog , Mem line_arena , u32 id , CircuitAddress addr)
{
	
	LineSpec * line = get_line_spec(line_arena , id);
	assert(addr.min || addr.max);
	assert(!line -> addr.min && !line -> addr.max);
	
	line -> addr = addr;
	name_addr(prog , line -> name , addr , 0);
}

void push_ctr_line(ControlUnitSpec * spec , u32 line_id)
{
	
	u32 * ctr_lines = spec -> ctr_lines;
	u32 count = spec -> ctr_line_count;
	u32 capacity = spec -> ctr_line_capacity;
	
	assert(count < capacity);
	ctr_lines[count++] = line_id;
	spec -> ctr_line_count = count;
}

InstrSpec * push_instr_spec(
							DataPathSpec * path , Mem * instr_arena , Mem * ctr_arena , u32 ctr_line_count , char * mnemonic , u32 op_code , u32 type)
{
	
	
	InstrSpec * instr_specs = mem_start(*instr_arena , InstrSpec);
	u32 count = mem_size(*instr_arena , InstrSpec);
	
	for(u32 i = 0; i < count; i++)
	{
		assert(instr_specs[i].op_code != op_code);
		assert(!str_eql(instr_specs[i].mnemonic , mnemonic));
	}
	
	InstrSpec * instr_spec = mem_push_struct(instr_arena , InstrSpec);
	*instr_spec = {};
	
	str_copy(instr_spec -> mnemonic , mnemonic);
	instr_spec -> type = type;
	instr_spec -> op_code = op_code;
	instr_spec -> ctr_line_ids = mem_push_array(ctr_arena , u32 , ctr_line_count);
	instr_spec -> ctr_line_values = mem_push_array(ctr_arena , u32 , ctr_line_count);
	instr_spec -> ctr_line_capacity = ctr_line_count;
	
	u32 path_instr_count = path -> instr_count;
	u32 path_instr_capacity = path -> instr_capacity;
	u32 * path_instr_indices = path -> instr_indices;
	
	assert(path_instr_count < path_instr_capacity);
	path_instr_indices[path_instr_count++] = count;
	path -> instr_count = path_instr_count;;
	
	return(instr_spec);
}

InstrSpec * get_instr_spec(Mem instr_arena , u32 op_code)
{
	
	
	InstrSpec * result = 0;
	InstrSpec * instrs = mem_start(instr_arena , InstrSpec);
	u32 count = mem_size(instr_arena , InstrSpec);
	
	for(u32 instr_index = 0; instr_index < count; instr_index++)
	{
		
		InstrSpec * instr = instrs + instr_index;
		if(instr -> op_code == op_code)
		{
			result = instr;
			break;
		}
	}
	
	return(result);
}

InstrSpec * get_instr_spec(Mem instr_arena , char * mnemonic)
{
	
	
	InstrSpec * result = 0;
	InstrSpec * instrs = mem_start(instr_arena , InstrSpec);
	u32 count = mem_size(instr_arena , InstrSpec);
	
	for(u32 instr_index = 0; instr_index < count; instr_index++)
	{
		
		InstrSpec * instr = instrs + instr_index;
		if(str_eql(instr -> mnemonic , mnemonic))
		{
			result = instr;
			break;
		}
	}
	
	return(result);
}

void set_ctr_line(InstrSpec * spec , u32 line_id , u32 line_value)
{
	
	u32 found = 0;
	u32 count = spec -> ctr_line_count;
	u32 capacity = spec -> ctr_line_capacity;
	u32 * ids = spec -> ctr_line_ids;
	u32 * values = spec -> ctr_line_values;
	
	for(u32 i = 0; i < count; i++)
	{
		if(spec -> ctr_line_ids[i] == line_id) 
		{
			found = 1;
			break;
		}
	}
	
	assert(!found);
	assert(count < capacity);
	
	ids[count] = line_id;
	values[count] = line_value;
	
	count++;
	spec -> ctr_line_count = count;
}


void push_stage_line_spec(
						  Mem * stage_arena , Mem * line_arena , 
						  u32 line_id , u32 stage_id , char * name , u32 width)
{
	
	push_line_spec(line_arena , line_id , name , width);
	
	StageLineSpec * stage_line = mem_push_struct(stage_arena , StageLineSpec);
	*stage_line = {};
	
	stage_line -> line_id = line_id;
	stage_line -> stage_id = stage_id;
	stage_line -> width = width;
	str_copy(stage_line -> name , name);
}

PortSpec * push_port_spec(
						  Mem * port_arena , Mem * line_arena , u32 line_id , char * name , char * filename , u32 direction , 
						  u32 width , u32 sign , u32 min , u32 max)
{
	
	//if(sign) 	{assert((i32)min < (i32)max);}
	//else 		{assert(min < max);}
	//assert(max-min <= 32);
	//assert(min < 32);
	//assert(max <= 32);
	
	push_line_spec(line_arena , line_id , name , width);
	PortSpec * port = mem_push_struct(port_arena , PortSpec);
	*port = {};
	
	port -> line_id = line_id;
	port -> width = width;
	port -> direction = direction;
	port -> sign = sign;
	port -> min = min;
	port -> max = max;
	
	str_copy(port -> name , name);
	str_copy(port -> filename , filename);
	
	return(port);
}


PortSpec * get_port_spec(Mem port_arena , u32 line_id)
{
	
	
	PortSpec * result = 0;
	PortSpec * ports = mem_start(port_arena , PortSpec);
	u32 port_count = mem_size(port_arena , PortSpec);
	
	for(u32 i = 0; i < port_count; i++)
	{
		
		PortSpec * port = ports + i;
		if(port -> line_id == line_id)
		{
			result = port;
			break;
		}
	}
	
	return(result);
}


RegisterSpec * push_register_spec(
								  Mem * reg_arena , Mem * line_arena , u32 type , u32 line_id , char * name , u32 width , u32 select_addr , u32 port_id)
{
	
	push_line_spec(line_arena , line_id , name , width);
	
	RegisterSpec * reg = mem_push_struct(reg_arena , RegisterSpec);
	*reg = {};
	
	reg -> type = type;
	reg -> line_id = line_id;
	reg -> select_addr = select_addr;
	reg -> port_id = port_id;
	str_copy(reg -> name , name);
	
	return(reg);
}

RegisterSpec * get_register_spec(Mem reg_arena , u32 addr)
{
	RegisterSpec * regs = mem_start(reg_arena , RegisterSpec);
	u32 count = mem_size(reg_arena , RegisterSpec);
	
	RegisterSpec * result = 0;
	for(u32 reg_index = 0; reg_index < count; reg_index++)
	{
		RegisterSpec * reg = regs + reg_index;
		
		if(reg -> select_addr == addr)
		{
			result = reg;
			break;
		}
	}
	
	return(result);
}

CircuitAddress get_reg_addr(Mem reg_arena , Mem line_arena , char * name)
{
	
	CircuitAddress result = {};
	RegisterSpec * registers = mem_start(reg_arena , RegisterSpec);
	u32 reg_count = mem_size(reg_arena , RegisterSpec);
	
	for(u32 reg_index = 0; reg_index < reg_count; reg_index++)
	{
		
		RegisterSpec * reg = registers + reg_index;
		if(str_eql(name , reg -> name))
		{
			result = get_line_addr(line_arena,  reg -> line_id);
			break;
		}
	}
	
	return(result);
}

u32 get_reg_select_addr(Mem reg_arena , Token * name_token)
{
	
	RegisterSpec * registers = mem_start(reg_arena , RegisterSpec);
	u32 reg_count = mem_size(reg_arena , RegisterSpec);
	
	u32 result = 0;
	u32 found = 0;
	
	if(name_token)
	{
		
		for(u32 reg_index = 0; reg_index < reg_count; reg_index++)
		{
			
			RegisterSpec * reg = registers + reg_index;
			if(token_eql(name_token , reg -> name))
			{
				result = reg -> select_addr;
				found = 1;
				break;
			}
		}
		
		if(!found)
		{
			
			char buffer[128] = {};
			copy_token_content(buffer , name_token);
			report_error("Unknown register %" , 1 , buffer);
		}
	}
	
	return(result);
}



InstrFieldSpec * push_instr_field_spec(
									   Mem * field_arena , Mem * line_arena , 
									   u32 type , u32 line_id , char * name , u32 min , u32 max , u32 sign)
{
	
	assert(min < max);
	assert(max-min <= 32);
	assert(min < 32);
	assert(max <= 32);
	
	push_line_spec(line_arena , line_id , name , max-min);
	
	InstrFieldSpec * field = mem_push_struct(field_arena , InstrFieldSpec);
	*field = {};
	
	str_copy(field -> name , name);
	field -> type = type;
	field -> line_id = line_id;
	field -> min = min;
	field -> max = max;
	field -> sign = sign;
	
	return(field);
}

InstrFieldSpec * get_instr_field_spec(Mem field_arena , u32 line_id)
{
	
	InstrFieldSpec * result = 0;
	InstrFieldSpec * fields = mem_start(field_arena , InstrFieldSpec);
	u32 count = mem_size(field_arena , InstrFieldSpec);
	
	for(u32 field_index = 0; field_index < count; field_index++)
	{
		InstrFieldSpec * field = fields + field_index;
		if(field -> line_id == line_id)
		{
			result = field;
			break;
		}
	}
	
	return(result);
}





DataPathSpec * push_data_path_spec(Mem * path_arena , u32 path_id , u32 instr_capacity , Mem * arena)
{
	
	DataPathSpec * paths = mem_start(*path_arena , DataPathSpec);
	u32 path_count = mem_size(*path_arena , DataPathSpec);
	
	for(u32 path_index = 0; path_index < path_count; path_index++)
	{
		DataPathSpec * curr_path = paths + path_index;
		assert(curr_path -> id != path_id);
	}
	
	DataPathSpec * path = mem_push_struct(path_arena , DataPathSpec);
	path -> id = path_id;
	path -> instr_indices = mem_push_array(arena , u32 , instr_capacity);
	path -> instr_capacity = instr_capacity;
	
	return(path);
}

DataPathSpec * get_data_path_from_op_code(Mem path_spec_arena , Mem instr_spec_arena,  u32 op_code)
{
	
	DataPathSpec * result = 0;
	DataPathSpec * paths = mem_start(path_spec_arena , DataPathSpec);
	u32 path_count = mem_size(path_spec_arena , DataPathSpec);
	
	InstrSpec * instrs = mem_start(instr_spec_arena , InstrSpec);
	u32 instr_count = mem_size(instr_spec_arena , InstrSpec);
	
	for(u32 path_index = 0; path_index < path_count; path_index++)
	{
		
		DataPathSpec * path = paths + path_index;
		
		u32 path_instr_count = path -> instr_count;
		u32 * path_instr_indices = path -> instr_indices;
		
		for(u32 path_instr_index = 0; path_instr_index < path_instr_count; path_instr_index++)
		{
			
			u32 instr_index = path_instr_indices[path_instr_index];
			InstrSpec * instr = instrs + instr_index;
			
			assert(instr_index < instr_count);
			
			if(instr -> op_code == op_code)
			{
				result = path;
				goto end;
			}
		}
	}
	
	end:
	
	return(result);
}






CircuitValue set_instr_bits(CircuitValue dest , CircuitValue src , u32 offset , u32 * mask_ptr)
{
	
	u32 min = offset;
	u32 max = src.width + offset;
	u32 value = src.v;
	
	assert(min < 32);
	assert(max <= 32);
	assert(max-min <= 32);
	
	assert(src.width <= dest.width);
	assert(min < dest.width);
	assert(max <= dest.width);
	
	u32 mask = get_bit_bound_mask(min , max);
	dest.v = (dest.v & ~mask) | ((value << min) & mask);
	
	u32 compare_mask = *mask_ptr & mask;
	assert(!compare_mask);
	
	*mask_ptr |= mask;
	
	return(dest);
	
}

CircuitValue field_to_cv(u32 value , InstrFieldSpec spec)
{
	assert(spec.min < spec.max);
	return(int_to_cv(value , spec.max - spec.min , spec.sign));
}

inline
u32 get_instr_fields(Mem field_arena , u32 instr_type , InstrFieldSpec * buffer)
{
	
	
	
	u32 count = 0;
	switch(instr_type)
	{
		case(INSTR_R_TYPE):
		{
			buffer[count++] = *get_instr_field_spec(field_arena , IR_OP_CODE_LINE);
			buffer[count++] = *get_instr_field_spec(field_arena , IR_RD_LINE);
			buffer[count++] = *get_instr_field_spec(field_arena , IR_RS_LINE);
			buffer[count++] = *get_instr_field_spec(field_arena , IR_RT_LINE);
			break;
		}
		
		case(INSTR_I_TYPE):
		{
			buffer[count++] = *get_instr_field_spec(field_arena , IR_OP_CODE_LINE);
			buffer[count++] = *get_instr_field_spec(field_arena , IR_RS_LINE);
			buffer[count++] = *get_instr_field_spec(field_arena , IR_RT_LINE);
			buffer[count++] = *get_instr_field_spec(field_arena , IR_IMM_LINE_DIRECT);
			break;
		}
		
		case(INSTR_J_TYPE):
		{
			buffer[count++] = *get_instr_field_spec(field_arena , IR_OP_CODE_LINE);
			buffer[count++] = *get_instr_field_spec(field_arena , IR_JUMP_LINE_DIRECT);
			break;
		}
		
	}
	
	return(count);
}

inline
u32 get_field_value(Instr instr , InstrFieldSpec field)
{
	
	u32 value = 0;
	u32 line_id = field.line_id;
	switch(line_id)
	{
		case(IR_OP_CODE_LINE): value = instr.op_code; break;
		case(IR_RD_LINE): value = instr.rd; break;
		case(IR_RS_LINE): value = instr.rs; break;
		case(IR_RT_LINE): value = instr.rt; break;
		case(IR_IMM_LINE_DIRECT): value = instr.imm; break;
		case(IR_JUMP_LINE_DIRECT): value = instr.pc_rel_jump; break;
		default: assert_zero;
	}
	
	return(value);
}



CircuitValue encode_instr(Mem field_arena , Instr * instr , u32 instr_type)
{
	
	CircuitValue result = {0 , 32};
	u32 overlap_mask = 0;
	
	InstrFieldSpec fields[32] = {};
	u32 field_count = get_instr_fields(field_arena , instr_type , fields);
	
	for(u32 field_index = 0; field_index < field_count; field_index++)
	{
		InstrFieldSpec field = fields[field_index];
		u32 value = get_field_value(*instr , field);
		CircuitValue cv = field_to_cv(value , field);
		result = set_instr_bits(result , cv , field.min , &overlap_mask);
	}
	
	return(result);
}

void interpret_MIPS(
					UninterpretedInstr * imm_instrs , u32 imm_instr_count , 
					Instr * instrs , u32 * instr_count_ptr , u32 instr_capacity , 
					Tag * tags , u32 * tag_count_ptr , u32 tag_capacity , 
					Mem reg_spec_arena , Mem instr_spec_arena)
{
	
	u32 instr_count = 0;
	u32 tag_count = 1;
	
	char mnemonic[64] = {};
	char r0[64] = {};
	char r1[64] = {};
	char r2[64] = {};
	char imm[64] = {};
	char tag_name[64] = {};
	
	for(u32 instr_index = 0; instr_index < imm_instr_count; instr_index++)
	{
		
		UninterpretedInstr * instr = imm_instrs + instr_index;
		
		Token * mnemonic_token = instr -> mnemonic;
		Token * r0_token = instr -> regs[0];
		Token * r1_token = instr -> regs[1];
		Token * r2_token = instr -> regs[2];
		Token * imm_token = instr -> imm;
		Token * tag_token = instr -> tag;
		
		if(mnemonic_token) copy_token_content(mnemonic , mnemonic_token);
		if(r0_token) copy_token_content(r0 , r0_token);
		if(r1_token) copy_token_content(r1 , r1_token);
		if(r2_token) copy_token_content(r2 , r2_token);
		if(imm_token) copy_token_content(imm , imm_token);
		if(tag_token) copy_token_content(tag_name , tag_token);
		
		InstrSpec * info = get_instr_spec(instr_spec_arena , mnemonic);
		u32 rd_value = 0;
		u32 rt_value = 0;
		u32 rs_value = 0;
		u32 imm_value = 0;
		u32 jump_value = 0;
		u32 op_code = 0;
		
		if(!mnemonic_token && tag_token)
		{
			assert(tag_count < tag_capacity);
			Tag * tag = tags + tag_count++;
			
			copy_token_content(tag -> name , tag_token);
			tag -> instr_index = instr_count;
		}
		
		else 
		{
			if(info)
			{
				switch(info -> type)
				{
					case(INSTR_R_TYPE):
					{
						
						if(r0_token && r1_token && r2_token)
						{
							op_code = info -> op_code;
							rd_value = get_reg_select_addr(reg_spec_arena , r0_token);
							rt_value = get_reg_select_addr(reg_spec_arena , r1_token);
							rs_value = get_reg_select_addr(reg_spec_arena , r2_token);
						}
						
						else
						{
							report_error("insufficient registers for R type instruction" , 1);
						}
						
						break;	
					}
					
					case(INSTR_I_TYPE):
					{
						
						if(imm_token)
						{
							if(r0_token && r1_token)
							{
								imm_value = imm_token -> i32_value;
								rt_value = get_reg_select_addr(reg_spec_arena , r0_token);
								rs_value = get_reg_select_addr(reg_spec_arena , r1_token);
								op_code = info -> op_code;
							}
							else
							{
								report_error("insufficient registers for I type instruction" , 1);
							}
						}
						
						else
						{
							report_error("Missiing immediate value for I type instruction" , 1);
						}
						
						break;	
					}
					
					case(INSTR_J_TYPE):
					{
						
						if(tag_token)
						{
							
							u32 found = 0;
							for(u32 tag_index = 0; tag_index < tag_count; tag_index++)
							{
								Tag * tag = tags + tag_index;
								if(token_eql(tag_token , tag -> name))
								{
									
									jump_value = tag_index;
									op_code = info -> op_code;
									found = 1;
									break;
								}
							}
							
							if(!found)
							{
								report_error("undefined tag" , 1);
							}
						}
						
						break;	
					}
					
					default:
					{
						assert_zero;
					}
				}
			}
			
			else
			{
				report_error("unknown mnemonic %" , 1 , mnemonic);
			}
			
			assert(instr_count < instr_capacity);
			Instr * new_instr = instrs + instr_count++;
			
			new_instr -> op_code = op_code;
			new_instr -> rd = rd_value;
			new_instr -> rt = rt_value;
			new_instr -> rs = rs_value;
			new_instr -> imm = imm_value;
			
			// NOTE: intermediate jump value. will be patched after
			new_instr -> pc_rel_jump = jump_value; 
		}
	}
	
	for(u32 instr_index = 0; instr_index < instr_count; instr_index++)
	{
		
		Instr * instr = instrs + instr_index;
		u32 jump_value = instr -> pc_rel_jump;
		
		if(jump_value)
		{
			Tag * tag = tags + jump_value;
			instr -> pc_rel_jump = (i32)tag -> instr_index - (i32)instr_index;
		}
	}
	
	
	*instr_count_ptr = instr_count;
	*tag_count_ptr = tag_count;	
	
}


void encode_MIPS(
				 Instr * instrs , u32 instr_count , 
				 Mem instr_spec_arena , Mem field_arena)
{
	
	
	for(u32 instr_index = 0; instr_index < instr_count; instr_index++)
	{
		Instr * curr_instr = instrs + instr_index;
		InstrSpec * info = get_instr_spec(instr_spec_arena , curr_instr -> op_code);
		curr_instr -> encoded_cv.width = 32;
		
		if(info)
		{
			
			u32 type = info -> type;
			curr_instr -> encoded_cv = encode_instr(field_arena , curr_instr , type);
		}
	}
}


void connect(CircuitProgram * prog , CircuitAddress from , CircuitAddress to)
{
	
	assert(width_of(from) == width_of(to));
	
	HardwareInstr * instr = push_hw_instruction(prog , HW_CONNECTION , 1 , 1);
	instr -> inputs[0] = from;
	instr -> outputs[0] = to;
	
}


HardwareInstr * push_transmission_gate(
									   CircuitProgram * prog , CircuitAddress Q_addr , 
									   CircuitAddress A_addr , CircuitAddress S_addr)
{
	HardwareInstr * instr = push_hw_instruction(prog , COMP_TRANS_GATE , 2 , 1);
	instr -> outputs[0] = Q_addr;
	instr -> inputs[0] = A_addr;
	instr -> inputs[1] = S_addr;
	
	assert(width_of(A_addr) == width_of(Q_addr));
	assert(width_of(S_addr) == 1);
	
	return(instr);
}

HardwareInstr * push_not(
						 CircuitProgram * prog , 
						 CircuitAddress Q_addr , 
						 CircuitAddress A_addr)
{
	
	HardwareInstr * instr = push_hw_instruction(prog , COMP_NOT , 1 , 1);
	instr -> outputs[0] = Q_addr;
	instr -> inputs[0] = A_addr;
	
	assert(width_of(A_addr) == width_of(Q_addr));
	return(instr);
}


HardwareInstr * push_and2(
						  CircuitProgram * prog , 
						  CircuitAddress Q_addr , 
						  CircuitAddress A_addr , CircuitAddress B_addr)
{
	
	HardwareInstr * instr = push_hw_instruction(prog , COMP_AND2 , 2 , 1);
	instr -> outputs[0] = Q_addr;
	instr -> inputs[0] = A_addr;
	instr -> inputs[1] = B_addr;
	
	assert(width_of(A_addr) == width_of(B_addr));
	assert(width_of(A_addr) == width_of(Q_addr));
	
	return(instr);
}


HardwareInstr * push_and4(
						  CircuitProgram * prog , 
						  CircuitAddress Q_addr , 
						  CircuitAddress A_addr , CircuitAddress B_addr , 
						  CircuitAddress C_addr , CircuitAddress D_addr)
{
	
	HardwareInstr * instr = push_hw_instruction(prog , COMP_AND4 , 4 , 1);
	instr -> outputs[0] = Q_addr;
	instr -> inputs[0] = A_addr;
	instr -> inputs[1] = B_addr;
	instr -> inputs[2] = C_addr;
	instr -> inputs[3] = D_addr;
	
	assert(width_of(A_addr) == width_of(B_addr));
	assert(width_of(A_addr) == width_of(C_addr));
	assert(width_of(A_addr) == width_of(D_addr));
	assert(width_of(A_addr) == width_of(Q_addr));
	
	return(instr);
}


void push_andN(
			   CircuitProgram * prog , 
			   CircuitAddress Q_addr , 
			   CircuitAddress * inputs , u32 count , Mem * arena)
{
	
	void * pop_addr = mem_end(*arena , void);
	u32 width = width_of(Q_addr);
	CircuitAddress * front_buffer = mem_push_array(arena , CircuitAddress , count);
	CircuitAddress * back_buffer = mem_push_array(arena , CircuitAddress , count);
	
	mem_copy(front_buffer , inputs , sizeof(CircuitAddress)*count);
	u32 front_buffer_size = count;
	u32 back_buffer_size = 0;
	
	while(front_buffer_size > 1)
	{
		u32 input_index = 0;
		
		while(input_index < front_buffer_size)
		{
			
			CircuitAddress next_output = {};
			
			if(input_index+4 <= front_buffer_size)
			{
				if(input_index+4 == front_buffer_size) 	next_output = Q_addr;
				else									next_output = push_addr(prog , width); 
				
				push_and4(
						  prog , next_output , 
						  front_buffer[input_index+0] , front_buffer[input_index+1] , 
						  front_buffer[input_index+2] , front_buffer[input_index+3]
						  );
				
				input_index += 4;
			}
			
			else if(input_index+2 <= front_buffer_size)
			{
				
				if(input_index+2 == front_buffer_size) 	next_output = Q_addr;
				else							next_output = push_addr(prog , width); 
				
				push_and2(
						  prog , next_output , 
						  front_buffer[input_index+0] , front_buffer[input_index+1]
						  );
				
				input_index += 2;
			}
			
			else
			{
				next_output = front_buffer[input_index++];
			}
			
			assert(back_buffer_size < count);
			back_buffer[back_buffer_size++] = next_output;
			
		}
		
		mem_copy(front_buffer , back_buffer , sizeof(CircuitAddress)*back_buffer_size);
		front_buffer_size = back_buffer_size;
		back_buffer_size = 0;
	}
	
	
	mem_pop(arena , pop_addr);
}


HardwareInstr * push_or2(
						 CircuitProgram * prog , 
						 CircuitAddress Q_addr , 
						 CircuitAddress A_addr , CircuitAddress B_addr)
{
	
	HardwareInstr * instr = push_hw_instruction(prog , COMP_OR2 , 2 , 1);
	instr -> outputs[0] = Q_addr;
	instr -> inputs[0] = A_addr;
	instr -> inputs[1] = B_addr;
	
	assert(width_of(A_addr) == width_of(B_addr));
	assert(width_of(A_addr) == width_of(Q_addr));
	
	return(instr);
}

HardwareInstr * push_or4(
						 CircuitProgram * prog , 
						 CircuitAddress Q_addr , 
						 CircuitAddress A_addr , CircuitAddress B_addr , 
						 CircuitAddress C_addr , CircuitAddress D_addr)
{
	
	HardwareInstr * instr = push_hw_instruction(prog , COMP_OR4 , 4 , 1);
	instr -> outputs[0] = Q_addr;
	instr -> inputs[0] = A_addr;
	instr -> inputs[1] = B_addr;
	instr -> inputs[2] = C_addr;
	instr -> inputs[3] = D_addr;
	
	assert(width_of(A_addr) == width_of(B_addr));
	assert(width_of(A_addr) == width_of(C_addr));
	assert(width_of(A_addr) == width_of(D_addr));
	assert(width_of(A_addr) == width_of(Q_addr));
	
	return(instr);
}



void push_orN(
			  CircuitProgram * prog , 
			  CircuitAddress Q_addr , 
			  CircuitAddress * inputs , u32 count , Mem * arena)
{
	
	void * pop_addr = mem_end(*arena , void);
	u32 width = width_of(Q_addr);
	CircuitAddress * front_buffer = mem_push_array(arena , CircuitAddress , count);
	CircuitAddress * back_buffer = mem_push_array(arena , CircuitAddress , count);
	
	mem_copy(front_buffer , inputs , sizeof(CircuitAddress)*count);
	u32 front_buffer_size = count;
	u32 back_buffer_size = 0;
	
	while(front_buffer_size > 1)
	{
		u32 input_index = 0;
		
		while(input_index < front_buffer_size)
		{
			
			CircuitAddress next_output = {};
			
			if(input_index+4 <= front_buffer_size)
			{
				if(input_index+4 == front_buffer_size) 	next_output = Q_addr;
				else									next_output = push_addr(prog , width); 
				
				push_or4(
						 prog , next_output , 
						 front_buffer[input_index+0] , front_buffer[input_index+1] , 
						 front_buffer[input_index+2] , front_buffer[input_index+3]
						 );
				
				input_index += 4;
			}
			
			else if(input_index+2 <= front_buffer_size)
			{
				
				if(input_index+2 == front_buffer_size) 	next_output = Q_addr;
				else							next_output = push_addr(prog , width); 
				
				push_or2(
						 prog , next_output , 
						 front_buffer[input_index+0] , front_buffer[input_index+1]
						 );
				
				input_index += 2;
			}
			
			else
			{
				next_output = front_buffer[input_index++];
			}
			
			assert(back_buffer_size < count);
			back_buffer[back_buffer_size++] = next_output;
			
		}
		
		mem_copy(front_buffer , back_buffer , sizeof(CircuitAddress)*back_buffer_size);
		front_buffer_size = back_buffer_size;
		back_buffer_size = 0;
	}
	
	
	mem_pop(arena , pop_addr);
}



HardwareInstr * push_adder(
						   CircuitProgram * prog , 
						   CircuitAddress Q_addr , 
						   CircuitAddress A_addr , CircuitAddress B_addr)
{
	
	HardwareInstr * instr = push_hw_instruction(prog , COMP_ADDER , 2 , 1);
	instr -> outputs[0] = Q_addr;
	instr -> inputs[0] = A_addr;
	instr -> inputs[1] = B_addr;
	
	assert(width_of(A_addr) == width_of(B_addr));
	assert(width_of(A_addr) == width_of(Q_addr));
	
	return(instr);
}

HardwareInstr * push_multiplier(
								CircuitProgram * prog , 
								CircuitAddress Q_addr , 
								CircuitAddress A_addr , CircuitAddress B_addr)
{
	
	HardwareInstr * instr = push_hw_instruction(prog , COMP_MULTIPLIER , 2 , 1);
	instr -> outputs[0] = Q_addr;
	instr -> inputs[0] = A_addr;
	instr -> inputs[1] = B_addr;
	
	assert(width_of(A_addr) == width_of(B_addr));
	assert(width_of(A_addr) == width_of(Q_addr));
	
	return(instr);
}





HardwareInstr * push_mux2(
						  CircuitProgram * prog , 
						  CircuitAddress Q_addr , 
						  CircuitAddress A_addr , CircuitAddress B_addr , CircuitAddress select_addr)
{
	
	HardwareInstr * instr = push_hw_instruction(prog , COMP_MUX2 , 3 , 1);
	instr -> outputs[0] = Q_addr;
	instr -> inputs[0] = A_addr;
	instr -> inputs[1] = B_addr;
	instr -> inputs[2] = select_addr;
	
	//instr -> min_bits[0] = 0;
	//instr -> max_bits[0] = 8;
	
	return(instr);
}

HardwareInstr * push_demux2(
							CircuitProgram * prog , 
							CircuitAddress Q0_addr , CircuitAddress Q1_addr , 
							CircuitAddress in_addr , CircuitAddress select_addr)
{
	HardwareInstr * instr = push_hw_instruction(prog , COMP_DEMUX2 , 2 , 2);
	instr -> outputs[0] = Q0_addr;
	instr -> outputs[1] = Q1_addr;
	instr -> inputs[0] = in_addr;
	instr -> inputs[1] = select_addr;
	
	return(instr);
}


HardwareInstr *  push_flip_flop(
								CircuitProgram * prog , 
								CircuitAddress Q_addr , 
								CircuitAddress D_addr , CircuitAddress clock_addr , 
								CircuitAddress chip_select)
{
	
	HardwareInstr * instr = push_hw_instruction(prog , COMP_FLIP_FLOP , 3 , 1);
	instr -> outputs[0] = Q_addr;
	instr -> inputs[0] = D_addr;
	instr -> inputs[1] = clock_addr;
	instr -> inputs[2] = chip_select;
	
	return(instr);
}

HardwareInstr *  push_flip_flop(
								CircuitProgram * prog , 
								CircuitAddress Q_addr , 
								CircuitAddress D_addr , CircuitAddress clock_addr)
{
	
	return(push_flip_flop(prog , Q_addr , D_addr , clock_addr , bit_sel(VDD , 0)));
}

void push_mux4(
			   CircuitProgram * prog ,
			   CircuitAddress out , 
			   CircuitAddress A , CircuitAddress B , CircuitAddress C , CircuitAddress D , CircuitAddress S)
{
	
	
	u32 width = width_of(out);
	assert(width_of(A) == width);
	assert(width_of(B) == width);
	assert(width_of(C) == width);
	assert(width_of(D) == width);
	assert(width_of(S) == 2);
	
	CircuitAddress S0 = bit_sel(S , 0);
	CircuitAddress S1 = bit_sel(S , 1);
	
	// mux immediate
	CircuitAddress mi0 = push_addr(prog , width);
	CircuitAddress mi1 = push_addr(prog , width);
	
	push_mux2(prog , mi0 , A , B , S0);
	push_mux2(prog , mi1 , C , D , S0);
	push_mux2(prog , out , mi0 , mi1 , S1);
}

void push_demux4(
				 CircuitProgram * prog , 
				 CircuitAddress Q0 , CircuitAddress Q1 , CircuitAddress Q2 , CircuitAddress Q3 , 
				 CircuitAddress input , CircuitAddress S)
{
	
	u32 width = width_of(input);
	assert(width_of(Q0) == width);
	assert(width_of(Q1) == width);
	assert(width_of(Q2) == width);
	assert(width_of(Q3) == width);
	assert(width_of(S) == 2);
	
	
	CircuitAddress S0 = bit_sel(S , 0);
	CircuitAddress S1 = bit_sel(S , 1);
	
	// demux immediate
	CircuitAddress di0 = push_addr(prog , width);
	CircuitAddress di1 = push_addr(prog , width);
	
	push_demux2(prog , di0 , di1 , input , S1);
	push_demux2(prog , Q0 , Q1 , di0 , S0);
	push_demux2(prog , Q2 , Q3 , di1 , S0);
}


void generate_muxN(
				   CircuitProgram * prog , Mem * arena , 
				   CircuitAddress output , 
				   CircuitAddress * inputs , CircuitAddress select, 
				   u32 input_count , u32 input_width)
{
	
	// NOTE: the most significant 1 in the input_count has the digit value equal to the number of outputs of the multiplexer.
	// It is equivilent to input_count/2 rounded up to the next power of 2.
	
	u32 width = width_of(select);
	u32 bsr = bitscan_right(input_count);
	assert(width == bsr);
	assert(input_count <= 1 << bsr);
	assert(input_count > 1 << (bsr-1));
	
	CircuitAddress * curr_inputs = mem_push_array(arena , CircuitAddress , input_count);
	mem_copy(curr_inputs,  inputs , sizeof(CircuitAddress)*input_count);
	
	for(u32 col = 0; col < width; col++)
	{
		CircuitAddress select_bit = bit_sel(select , col);
		for(u32 row = 0; row*2 < input_count; row++)
		{
			
			u32 index = row*2;
			CircuitAddress inputA = curr_inputs[index+0];
			CircuitAddress inputB = curr_inputs[index+1];
			CircuitAddress new_input_addr = inputA;
			
			assert(width_of(inputA) == input_width);
			assert(width_of(inputB) == input_width);
			
			if(index+1 < input_count)
			{
				if(input_count == 2)	new_input_addr = output;
				else					new_input_addr = push_addr(prog , input_width); 
				
				push_mux2(prog , new_input_addr , inputA , inputB , select_bit);
			}
			
			curr_inputs[row] = new_input_addr;
		}
		
		input_count >>= 1;
	}
	
	assert(input_count == 1);
	
	mem_pop(arena , curr_inputs);
}



void generate_demuxN(
					 CircuitProgram * prog , Mem * arena , 
					 CircuitAddress * outputs , u32 output_count , 
					 CircuitAddress input , CircuitAddress select)
{
	
	// NOTE: the most significant 1 in the input_count has the digit value equal to the number of outputs of the multiplexer.
	// It is equivilent to input_count/2 rounded up to the next power of 2.
	
	u32 width = width_of(select);
	u32 bsr = bitscan_right(output_count);
	assert(width == bsr);
	assert(output_count <= 1 << bsr);
	assert(output_count > 1 << (bsr-1));
	
	u32 buffer_size = 1;
	CircuitAddress * buffer0 = mem_push_array(arena , CircuitAddress , output_count);
	CircuitAddress * buffer1 = mem_push_array(arena , CircuitAddress , output_count);
	CircuitAddress * front_buffer = buffer0;
	CircuitAddress * back_buffer = buffer1;
	
	front_buffer[0] = input;
	
	for(u32 col = 0; col < width; col++)
	{
		
		assert(front_buffer != back_buffer);
		CircuitAddress select_bit = bit_sel(select , col);
		for(u32 row = 0; row < buffer_size; row++)
		{
			
			u32 dest_index = row*2;
			assert(dest_index+0 < output_count);
			assert(dest_index+1 < output_count);
			
			CircuitAddress sub_input = front_buffer[row];
			CircuitAddress output_A = {};
			CircuitAddress output_B = {};
			
			//back_buffer[dest_index+0];
			if(col+1 < width)
			{
				output_A = push_addr(prog , 1); 
				output_B = push_addr(prog , 1); 
			}
			
			else
			{
				output_A = outputs[dest_index+0];
				output_B = outputs[dest_index+1];
			}
			
			push_demux2(prog , output_A , output_B , sub_input , select_bit);
			back_buffer[dest_index+0] = output_A;
			back_buffer[dest_index+1] = output_B;
		}
		
		front_buffer = (front_buffer == buffer0) ? buffer1 : buffer0;
		back_buffer = (back_buffer == buffer0) ? buffer1 : buffer0;
		buffer_size <<= 1;
	}
	
	mem_pop(arena , front_buffer);
}

CircuitAddress get_op_bit_input_line(CircuitAddress * bits , CircuitAddress * inv_bits , u32 index , u32 op_code)
{
	u32 bit_value = (op_code >> index) & 0x1;
	CircuitAddress result = bit_value ? bits[index] : inv_bits[index];
	
	return(result);
}


void push_sign_ext(
				   CircuitProgram * prog , MipsSpec * spec , 
				   CircuitAddress out , CircuitAddress in , u32 sign)
{
	
	u32 in_width = width_of(in);
	u32 out_width = width_of(out);
	
	if(sign)
	{
		if(in_width >= out_width)
		{
			connect(prog , sub_addr(in , 0 , out_width) , out);
		}
		else
		{
			connect(prog , in , sub_addr(out , 0 , in_width));
			
			// NOTE: sign extension
			for(u32 i = in_width; i < out_width; i++)
			{
				connect(prog , sub_addr(in , in_width-1 , in_width) , sub_addr(out , i , i+1));
			}
			
		}
	}
	
	else
	{
		// TODO: unsigned sign ext
		assert_zero;
	}
}


void MIPS_generate_stage_counter(
								 CircuitProgram * prog , Mem stage_arena , Mem line_arena , 
								 CircuitAddress clock , Mem * arena)
{
	
	StageLineSpec * stage_lines = mem_start(stage_arena , StageLineSpec);
	u32 stage_count = mem_size(stage_arena , StageLineSpec);
	
	CircuitAddress * stage_line_addrs = mem_push_array(arena , CircuitAddress , stage_count);
	for(u32 stage_index = 0; stage_index < stage_count; stage_index++)
	{
		
		CircuitAddress addr = get_line_addr(line_arena , stage_lines[stage_index].line_id);
		stage_line_addrs[stage_index] = addr;
	}
	
	for(u32 reg_index = 0; reg_index < stage_count; reg_index++)
	{
		
		u32 index1 = reg_index + 0;
		u32 index2 = reg_index + 1;
		if(index2 >= stage_count) index2 = 0;
		
		CircuitAddress curr_line = stage_line_addrs[index1];
		CircuitAddress next_line = stage_line_addrs[index2];
		
		if(index2 == 0)
		{
			CircuitAddress imm1 = push_addr(prog , 1);
			CircuitAddress imm2 = push_addr(prog , 1);
			CircuitAddress or_reset = push_addr(prog , 1);
			
			push_orN(prog , imm1 , stage_line_addrs , stage_count , arena);
			push_not(prog , imm2 , imm1);
			push_or2(prog , or_reset , curr_line , imm2);
			
			curr_line = or_reset;
			
			
		}
		
		push_flip_flop(prog , next_line , curr_line , clock);
	}
}


void push_load_reg_tests(
						 CircuitProgram * prog ,
						 CircuitAddress * registers , u32 reg_count , 
						 CircuitAddress reg_addr , CircuitAddress output)
{
	
	
	TestVector * load_test = push_test_vec(prog , 1 , 0);
	
	push_conseq(
				make_expr(
						  output , 
						  make_expr(reg_addr , registers[0] , EXPR_MEMORY , {-2} , {-2} , prog) , 
						  EXPR_EQL , {0} , prog) , 
				load_test , prog
				);
	
}


void push_store_reg_tests(
						  CircuitProgram * prog , 
						  CircuitAddress r0 , CircuitAddress r1 , CircuitAddress r2 , CircuitAddress r3 , 
						  CircuitAddress store_data_addr , CircuitAddress store_addr)
{
	
	
	TestVector * store_test1 = push_test_vec(prog , 1 , 1);
	push_conseq(
				make_expr(r0 , store_data_addr , EXPR_EQL , {0} , {0} , prog) , 
				store_test1 , prog
				);
	
	push_antec(
			   make_expr(store_addr , (u32)0 , EXPR_EQL , {0} , {0} , prog) , 
			   store_test1 , prog
			   );
	
	
	TestVector * store_test2 = push_test_vec(prog , 1 , 1);
	push_conseq(
				make_expr(r1 , store_data_addr , EXPR_EQL , {0} , {0} , prog) , 
				store_test2 , prog
				);
	
	push_antec(
			   make_expr(store_addr , 1 , EXPR_EQL , {0} , {0} , prog) , 
			   store_test2 , prog
			   );
	
	
	TestVector * store_test3 = push_test_vec(prog , 1 , 1);
	push_conseq(
				make_expr(r2 , store_data_addr , EXPR_EQL , {0} , {0} , prog) , 
				store_test3 , prog
				);
	
	push_antec(
			   make_expr(store_addr , 2 , EXPR_EQL , {0} , {0} , prog) , 
			   store_test3 , prog
			   );
	
	
	
	TestVector * store_test4 = push_test_vec(prog , 1 , 1);
	push_conseq(
				make_expr(r3 , store_data_addr , EXPR_EQL , {0} , {0} , prog) , 
				store_test4 , prog
				);
	
	push_antec(
			   make_expr(store_addr , 3 , EXPR_EQL , {0} , {0} , prog) , 
			   store_test4 , prog
			   );
}

inline
void extract_addr_lines(
						CircuitProgram * prog , 
						CircuitAddress addr , 
						CircuitAddress * pos_bits , 
						CircuitAddress * neg_bits)
{
	
	u32 addr_width = width_of(addr);
	CircuitAddress addr_inv = push_addr(prog , addr_width);
	push_not(prog , addr_inv , addr);
	
	for(u32 bit_index = 0; bit_index < addr_width; bit_index++)
	{
		pos_bits[bit_index] = bit_sel(addr , bit_index);
		neg_bits[bit_index] = bit_sel(addr_inv , bit_index);
	}
	
	
}

inline
void get_select_addr_lines(
						   u32 addr_value , u32 addr_width , 
						   CircuitAddress * select_bits , 
						   CircuitAddress * pos_bits , CircuitAddress * neg_bits)
{
	
	for(u32 bit_index = 0; bit_index < addr_width; bit_index++)
	{
		u32 bit_value = (addr_value >> bit_index) & 0x1;
		select_bits[bit_index] = bit_value ? pos_bits[bit_index] : neg_bits[bit_index];
	}
}

void generate_load_decoder(
						   CircuitProgram * prog , 
						   Mem line_arena , Mem reg_arena , 
						   CircuitAddress output , CircuitAddress select , 
						   Mem * arena)
{
	
	RegisterSpec * reg_specs = mem_start(reg_arena , RegisterSpec);
	u32 reg_count = mem_size(reg_arena , RegisterSpec);
	
	u32 select_width = width_of(select);
	CircuitAddress * select_bits = mem_push_array(arena , CircuitAddress , select_width);	
	CircuitAddress * pos_bits = mem_push_array(arena , CircuitAddress , select_width);
	CircuitAddress * neg_bits = mem_push_array(arena , CircuitAddress , select_width);
	
	extract_addr_lines(prog , select , pos_bits , neg_bits);
	
	for(u32 reg_index = 0; reg_index < reg_count; reg_index++)
	{
		
		RegisterSpec * reg_spec = reg_specs + reg_index;
		u32 reg_select_addr = reg_spec -> select_addr;
		CircuitAddress reg_circuit_addr = get_line_addr(line_arena , reg_spec -> line_id);
		CircuitAddress output_enable = push_addr(prog , 1);
		
		switch(reg_spec -> type)
		{
			
			case(ZR_REGISTER):
			case(IO_REGISTER):
			case(GP_REGISTER):
			{
				get_select_addr_lines(
									  reg_select_addr , select_width , 
									  select_bits , pos_bits , neg_bits
									  );
				
				push_andN(
						  prog , 
						  output_enable , 
						  select_bits , select_width , arena
						  );
				
				push_transmission_gate(prog , output , reg_circuit_addr , output_enable);
				break;
			}
			
			default:
			{
				assert_zero;
			}
		}
	}
}


#if 0
{
	CircuitAddress * load_A_pos_bits = mem_push_array(arena , CircuitAddress , reg_addr_width);
	CircuitAddress * load_A_neg_bits = mem_push_array(arena , CircuitAddress , reg_addr_width);
	
	CircuitAddress * load_B_pos_bits = mem_push_array(arena , CircuitAddress , reg_addr_width);
	CircuitAddress * load_B_neg_bits = mem_push_array(arena , CircuitAddress , reg_addr_width);
	
	extract_addr_lines(prog , load_addr_A , load_A_pos_bits , load_A_neg_bits);
	extract_addr_lines(prog , load_addr_B, load_B_pos_bits , load_B_neg_bits);
	
	push_module(prog , "loadA_mux");
	
	generate_load_decoder(
						  prog , line_arena , reg_arena , reg_addr_width , output_A , 
						  select_bits , load_A_pos_bits , load_A_neg_bits , arena
						  );
	
	pop_module(prog);
	
	push_module(prog , "loadB_mux");
	
	
	generate_load_decoder(
						  prog , line_arena , reg_arena , reg_addr_width , output_B , 
						  select_bits , load_B_pos_bits , load_B_neg_bits , arena
						  );
	
	pop_module(prog);
	
	
}
#endif


void MIPS_generate_register_file(
								 CircuitProgram * prog , Mem * arena , 
								 Mem line_arena , Mem reg_arena , Mem port_arena , 
								 u32 reg_width , CircuitAddress * regs , // TODO: fill regs
								 CircuitAddress clock , CircuitAddress chip_select , CircuitAddress reg_write_enable , 
								 CircuitAddress write_addr , CircuitAddress write_data)
{
	
	
	
	RegisterSpec * reg_specs = mem_start(reg_arena , RegisterSpec);
	u32 reg_count = mem_size(reg_arena , RegisterSpec);
	
	CircuitAddress write_enable = push_addr(prog , 1);
	push_and2(prog , write_enable , chip_select , reg_write_enable);
	
	void * pop_addr = mem_end(*arena , void);
	assert(width_of(write_data) == reg_width);
	
	u32 write_addr_width = width_of(write_addr);
	CircuitAddress * write_pos_bits = mem_push_array(arena , CircuitAddress , write_addr_width);
	CircuitAddress * write_neg_bits = mem_push_array(arena , CircuitAddress , write_addr_width);
	CircuitAddress * select_bits = mem_push_array(arena , CircuitAddress , write_addr_width);
	
	extract_addr_lines(prog , write_addr , write_pos_bits , write_neg_bits);
	
	push_module(prog , "registers");
	for(u32 reg_index = 0; reg_index < reg_count; reg_index++)
	{
		
		RegisterSpec * reg_spec = reg_specs + reg_index;
		CircuitAddress addr = {};
		
		switch(reg_spec -> type)
		{
			case(ZR_REGISTER):
			{
				
				addr = get_line_addr(line_arena , reg_spec -> line_id);
				connect(prog , sub_addr(GND , 0 , reg_width) , addr);
				break;
			}
			
			case(GP_REGISTER):
			{
				
				//get_line_addr(prog , line_arena , reg_spec -> line_id , reg_width);
				break;
			}
			
			case(IO_REGISTER):
			{
				
				//get_line_addr(prog , line_arena , reg_spec -> line_id , reg_width);
				break;
			}
			
			default:
			{
				assert_zero;
			}
		}
	}
	pop_module(prog);
	
	push_module(prog , "store_decoder");
	for(u32 reg_index = 0; reg_index < reg_count; reg_index++)
	{
		
		CircuitAddress addr_select = push_addr(prog , 1);
		CircuitAddress reg_enable = push_addr(prog , 1);
		RegisterSpec * reg_spec = reg_specs + reg_index;
		
		u32 reg_select_addr = reg_spec -> select_addr;
		CircuitAddress reg_addr = get_line_addr(line_arena , reg_spec -> line_id);
		
		push_and2(
				  prog , 
				  reg_enable , 
				  addr_select , write_enable
				  );
		
		switch(reg_spec -> type)
		{
			case(ZR_REGISTER):
			{
				break;
			}
			
			case(GP_REGISTER):
			{
				get_select_addr_lines(
									  reg_select_addr , write_addr_width , 
									  select_bits , 
									  write_pos_bits , write_neg_bits
									  );
				
				push_andN(
						  prog , 
						  addr_select , 
						  select_bits , write_addr_width , arena
						  );
				
				push_flip_flop(
							   prog , reg_addr , 
							   write_data , clock , reg_enable
							   );
				
				break;
			}
			
			case(IO_REGISTER):
			{
				
				assert_zero;
				PortSpec * port = get_port_spec(port_arena , reg_spec -> port_id);
				CircuitAddress port_addr = get_line_addr(line_arena , port -> line_id);
				CircuitAddress input_addr = write_data;
				
				if(!port -> direction)
				{
					input_addr = port_addr;
				}
				
				get_select_addr_lines(
									  reg_select_addr , write_addr_width , 
									  select_bits , 
									  write_pos_bits , write_neg_bits
									  );
				
				push_andN(
						  prog , 
						  addr_select , 
						  select_bits , write_addr_width , arena
						  );
				
				push_flip_flop(
							   prog , reg_addr , 
							   input_addr , clock , reg_enable
							   );
				
				break;
			}
			
			default:
			{
				assert_zero;
			}
		}
	}
	
	pop_module(prog);
	mem_pop(arena , pop_addr);
	
}

void push_alu_tests(CircuitProgram * prog , CircuitAddress adder_out , CircuitAddress input_A , CircuitAddress input_B)
{
	TestVector * adder_test = push_test_vec(prog , 1 , 2);
	
	push_conseq(
				make_expr(
						  adder_out , 
						  make_expr(input_A , input_B , EXPR_ADD , {-1} , {-1} , prog) ,
						  EXPR_EQL , {0} , prog
						  ) , adder_test , prog
				);
}


void MIPS_generate_ALU(
					   CircuitProgram * prog , u32 reg_width , 
					   CircuitAddress clock_addr ,
					   CircuitAddress alu_out , CircuitAddress adder_out , CircuitAddress mult_out , 
					   CircuitAddress shift_left_out , CircuitAddress shift_right_out , 
					   CircuitAddress input_A , CircuitAddress input_B , CircuitAddress func)
{
	
	HardwareInstr * adder = push_hw_instruction(prog , COMP_ADDER , 2 , 1);
	HardwareInstr * mult = push_hw_instruction(prog , COMP_MULTIPLIER , 2 , 1);
	HardwareInstr * shift_left = push_hw_instruction(prog , COMP_SHIFT_LEFT , 2 , 1);
	HardwareInstr * shift_right = push_hw_instruction(prog , COMP_SHIFT_RIGHT , 2 , 1);
	
	adder -> inputs[0] = input_A;
	adder -> inputs[1] = input_B;
	adder -> outputs[0] = adder_out;
	
	mult -> inputs[0] = input_A;
	mult -> inputs[1] = input_B;
	mult -> outputs[0] = mult_out;
	
	shift_left -> inputs[0] = input_A;
	shift_left -> inputs[1] = input_B;
	shift_left -> outputs[0] = shift_left_out;
	
	shift_right -> inputs[0] = input_A;
	shift_right -> inputs[1] = input_B;
	shift_right -> outputs[0] = shift_right_out;
	
	push_mux4(
			  prog , alu_out , adder_out , mult_out , shift_left_out , shift_right_out , func
			  );
}

void MIPS_generate_PC_logic(
							CircuitProgram * prog , CircuitAddress clock , CircuitAddress chip_select , CircuitAddress pc_write_enable , 
							CircuitAddress PC , CircuitAddress jump , CircuitAddress branch)
{
	
	u32 PC_width = width_of(PC);
	assert(PC_width);
	
	CircuitAddress enable = push_addr(prog , 1);
	push_and2(prog , enable , chip_select , pc_write_enable);
	
	CircuitAddress incr = push_addr(prog , "PC_incr" , PC_width , 0);
	CircuitAddress add_operand = push_addr(prog , "PC_add_operand" , PC_width , 0);
	CircuitAddress next = push_addr(prog , "PC_next" , PC_width , 0);
	
	connect(prog , sub_addr(VDD , 0 , 1) , sub_addr(incr , 0 , 1));
	connect(prog , sub_addr(GND , 1 , PC_width-1) , sub_addr(incr , 1 , PC_width-1));
	
	push_mux2(prog , add_operand , incr , jump , branch);
	push_adder(prog , next , PC , add_operand);
	push_flip_flop(prog , PC , next , clock , enable);
}			

#if 0	
void MIPS_generate_control_unit_tests(
									  CircuitProgram * prog , CircuitAddress op_code_addr , 
									  CircuitAddress * critical_path , u32 critical_path_length)
{
	CircuitAddress * control_bits = instr_set -> control_bits;
	u32 control_bit_count = instr_set -> control_bit_count;
	TraceBounds trace = {};
	
	u32 op_code_width = width_of(op_code_addr);
	
#if 0
	u32 max_found = 0;
	
	for( u32 control_bit_index = 0; 
		control_bit_index < control_bit_count; 
		control_bit_index++)
	{
		CircuitAddress bit_addr = control_bits[control_bit_index];
		
		u32 length = find_path_length(op_code_addr , bit_addr);
		if(length >= max_length)
		{
			max_length = length;
			critical_path_end = bit_addr;
			max_found = 1;
		}
	}
#endif
	
	if(critical_path_length >= 2)
	{
		trace = make_backtrace(critical_path , critical_path_length , prog);
	}
	
	for( u32 instr_spec_index = 0; 
		instr_spec_index < instr_spec_count; 
		instr_spec_index++)
	{
		
		InstrSpec * curr_spec = instr_specs + instr_spec_index;
		CircuitValue op_code = {curr_spec -> op_code , op_code_width};
		TestVector * test = push_test_vec(prog , control_bit_count , 2);
		
		push_antec(
				   make_expr(op_code_addr , op_code , EXPR_EQL , trace.start , {0}  , prog) , 
				   test , prog
				   );
		
		for( u32 control_bit_index = 0; 
			control_bit_index < control_bit_count; 
			control_bit_index++)
		{
			
			CircuitAddress bit_addr = control_bits[control_bit_index];
			u32 bit_value = curr_spec -> control_bit_values[control_bit_index];
			
			push_conseq(
						make_expr(bit_addr , bit_value , EXPR_EQL , trace.end , {0} , prog) , 
						test , prog
						);
		}
	}
	
}
#endif

struct ControlTruthTable
{
	CircuitAddress * ctr_bit_addrs;
	u32 * op_codes;
	u32 * truth_table;
	
	u32 row_count; // NOTE; ctr_bit_count == row_count
	u32 col_count; // NOTE: op_code_count = row_count;
};

ControlTruthTable get_control_line_truth_table(
											   Mem * arena , Mem line_arena , Mem instr_arena ,
											   u32 instr_start , u32 instr_end , u32 * ctr_lines , u32 ctr_line_count)
{
	
	InstrSpec * instrs = mem_start(instr_arena , InstrSpec);
	u32 instr_count = mem_size(instr_arena , InstrSpec);
	
	CircuitAddress * ctr_bit_addrs = 0;
	u32 * op_codes = 0;
	u32 * truth_table = 0;
	
	u32 col_count = 0;
	u32 row_count = 0;
	
	for(u32 line_index = 0; line_index < ctr_line_count; line_index++) 
	{
		CircuitAddress addr = get_line_addr(line_arena , ctr_lines[line_index]);
		col_count += width_of(addr);
	}
	row_count = instr_count;
	
	ctr_bit_addrs = mem_push_array(arena , CircuitAddress , col_count);
	op_codes = mem_push_array(arena , u32 , row_count);
	truth_table = mem_push_array(arena , u32 , col_count*row_count);
	
	u32 col_counter = 0;
	for(u32 line_index = 0; line_index < ctr_line_count; line_index++)
	{
		
		u32 curr_line = ctr_lines[line_index];
		CircuitAddress line_addr = get_line_addr(line_arena , curr_line);
		u32 width = width_of(line_addr);
		
		for(u32 bit_index = 0; bit_index < width; bit_index++)
		{
			
			CircuitAddress bit_addr = bit_sel(line_addr , bit_index);
			assert(col_counter < col_count);
			ctr_bit_addrs[col_counter++] = bit_addr;
		}
	}
	
	for(u32 instr_index = 0; instr_index < row_count; instr_index++)
	{
		op_codes[instr_index] = instrs[instr_index].op_code;
	}
	
	for(u32 row_index = 0; row_index < row_count; row_index++)
	{
		
		InstrSpec * instr = instrs + row_index;
		u32 * line_ids = instr -> ctr_line_ids;
		u32 * line_values = instr -> ctr_line_values;
		u32 line_count = instr -> ctr_line_count;
		
		for(u32 line_index = 0; line_index < line_count; line_index++)
		{
			
			u32 line_found = 0;
			u32 id = line_ids[line_index];
			u32 value = line_values[line_index];
			CircuitAddress line_addr = {};
			
			for(u32 ctr_line_index = 0; ctr_line_index < ctr_line_count; ctr_line_index++)
			{
				
				u32 curr_line = ctr_lines[ctr_line_index];
				if(id == curr_line)
				{
					line_addr = get_line_addr(line_arena , curr_line);
					line_found = 1;
					break;
				}
			}
			
			assert(line_count);
			u32 width = width_of(line_addr);
			
			for(u32 bit_index = 0; bit_index < width; bit_index++)
			{
				
				CircuitAddress bit_addr = bit_sel(line_addr , bit_index);
				u32 bit_value = (value >> bit_index) & 0x1;
				u32 bit_found = 0;
				
				for(u32 col_index = 0; col_index < col_count; col_index++)
				{
					if(ctr_bit_addrs[col_index] == bit_addr)
					{
						truth_table[row_index*col_count + col_index] = bit_value;
						bit_found = 1;
						break;
					}
				}
				
				assert(bit_found);
			}
		}
	}
	
	ControlTruthTable result;
	
	result.ctr_bit_addrs = ctr_bit_addrs;
	result.op_codes = op_codes;
	result.truth_table = truth_table;
	result.row_count = row_count;
	result.col_count = col_count;
	
	return(result);
}

void MIPS_generate_control_unit(
								CircuitProgram * prog , Mem * arena , 
								Mem line_arena , Mem instr_arena , u32 instr_start , u32 instr_end , 
								ControlUnitSpec * ctr_unit , 
								CircuitAddress clock , CircuitAddress op_code_addr)
{
	
	u32 op_code_width = width_of(op_code_addr);
	
	u32 * ctr_lines = ctr_unit -> ctr_lines;
	u32 ctr_line_count = ctr_unit -> ctr_line_count;
	
	CircuitAddress * op_code_bits = mem_push_array(arena , CircuitAddress , op_code_width);
	CircuitAddress * inv_op_code_bits = mem_push_array(arena , CircuitAddress , op_code_width);
	
	u32 driver_op_capacity = 64;
	u32 driver_op_count = 0;
	u32 * driver_ops = mem_push_array(arena , u32 , driver_op_capacity);
	CircuitAddress * and_outputs = mem_push_array(arena , CircuitAddress , driver_op_capacity);
	
	//u32 start_instr_index = prog -> hw_instr_count;
	//u32 end_instr_index = prog -> hw_instr_count;
	
	// NOTE: generating input lines from op code.
	
	for(u32 op_bit_index = 0; op_bit_index < op_code_width; op_bit_index++)
	{
		op_code_bits[op_bit_index] = bit_sel(op_code_addr , op_bit_index);
		inv_op_code_bits[op_bit_index] = push_addr(prog , 1);
		push_not(prog , inv_op_code_bits[op_bit_index] , op_code_bits[op_bit_index]);
	}
	
	ControlTruthTable truth_table_result = get_control_line_truth_table(
																		arena , line_arena , instr_arena , instr_start , instr_end , ctr_lines , ctr_line_count
																		);
	
	CircuitAddress * control_bits = truth_table_result.ctr_bit_addrs;
	u32 * op_codes = truth_table_result.op_codes;
	u32 * truth_table = truth_table_result.truth_table;
	
	u32 control_bit_count = truth_table_result.col_count;
	u32 op_code_count = truth_table_result.row_count;
	
	// NOTE: generate PLA structure.
	
	for( u32 control_bit_index = 0; 
		control_bit_index < control_bit_count; 
		control_bit_index++)
	{
		
		CircuitAddress bit_addr = control_bits[control_bit_index];
		driver_op_count = 0;
		and_outputs[0] = sub_addr(GND , 0 , 1);
		
		// NOTE: collecting the op codes that line_arena the current control bit.
		for(u32 op_code_index = 0; op_code_index < op_code_count; op_code_index++)
		{
			
			u32 op_code = op_codes[op_code_index];
			u32 bit_value = truth_table[op_code_index*control_bit_count + control_bit_index];
			
			if(bit_value)
			{
				assert(driver_op_count < driver_op_capacity);
				driver_ops[driver_op_count++] = op_code;
			}
		}
		
		// NOTE: generating AND circuit for each collected op code, yeilding select lines.
		for(u32 driver_index = 0; driver_index < driver_op_count; driver_index++)
		{
			
			u32 op_code = driver_ops[driver_index];
			CircuitAddress last_output = get_op_bit_input_line(op_code_bits , inv_op_code_bits , 0 , op_code);
			
			for(u32 op_bit_index = 1; op_bit_index < op_code_width; op_bit_index++)
			{
				CircuitAddress op_bit_addr = get_op_bit_input_line(op_code_bits , inv_op_code_bits , op_bit_index , op_code);
				CircuitAddress next_output = push_addr(prog , 1);
				
				push_and2(prog , next_output , last_output , op_bit_addr);
				last_output = next_output;
			}
			
			and_outputs[driver_index] = last_output;
		}
		
		// NOTE: generating OR circuit to collate all select lines to final output.
		CircuitAddress last_output = and_outputs[0];
		
		for(u32 and_index = 1; and_index < driver_op_count; and_index++)
		{
			
			CircuitAddress curr_and_addr = and_outputs[and_index];
			CircuitAddress next_output = push_addr(prog , 1);
			
			push_or2(prog , next_output , last_output , curr_and_addr);
			last_output = next_output;
		}
		
		push_flip_flop(prog , bit_addr , last_output , clock);
	}
	
#if 0
	end_instr_index = prog -> hw_instr_count;
	
	CircuitAddress * critical_path = 0;
	
	u32 critical_path_length = find_critical_path(
												  op_code_addr , control_bits[0] , 
												  prog -> hw_instrs , start_instr_index , end_instr_index , 
												  &critical_path , arena
												  );
	
	//MIPS_generate_control_unit_tests(prog , instr_set , op_code_addr , critical_path , critical_path_length);
#endif
	
	ctr_unit -> ctr_bit_addrs = control_bits;
	ctr_unit -> op_codes = op_codes;
	ctr_unit -> ctr_truth_table = truth_table;
	
	ctr_unit -> ctr_bit_count = control_bit_count;
	ctr_unit -> op_code_count = op_code_count;
}


void set_machine_instr_bits(CircuitProgram * prog , CircuitAddress addr , CircuitAddress rel_sub_addr , u32 value)
{
	
	u32 width = width_of(addr);
	assert(rel_sub_addr.min <= width && rel_sub_addr.max <= width);
	
	CircuitAddress sub_addr = {addr.min + rel_sub_addr.min , addr.min + rel_sub_addr.max};
	push_input_vector(prog , sub_addr , value);
}


inline
CircuitAddress rel_addr(CircuitAddress child , CircuitAddress parent)
{
	assert(parent.min <= child.min && child.min <= parent.max);
	assert(parent.min <= child.max && child.max <= parent.max);
	
	CircuitAddress result = {child.min - parent.min , child.max - parent.min};
	return(result);
}


void MIPS_generate_instruction_memory(
									  CircuitProgram * prog , Instr * sw_instr , u32 sw_instr_count , 
									  CircuitAddress clock , CircuitAddress chip_select , 
									  CircuitAddress IR  , CircuitAddress PC)
{
	
	
	u32 machine_width = width_of(IR);
	CircuitAddress first_addr = {};
	CircuitAddress last_addr = {};
	u32 first_address_allocated = 0;
	
	for(u32 instr_index = 0; instr_index < sw_instr_count; instr_index++)
	{
		Instr * curr_instr = sw_instr + instr_index;
		
		char name[128] = {};
		str_format(name , "i%" , instr_index);
		CircuitAddress machine_addr = push_addr(prog , name , machine_width , 0);
		
		push_input_vector(prog , machine_addr , curr_instr -> encoded_cv);
		
		if(!first_address_allocated)
		{
			first_addr = machine_addr;
			first_address_allocated = 1;
		}
		
		last_addr = machine_addr;
	}
	
	
	CircuitAddress mem_out_addr = push_addr(prog , "instr-mem-out" , machine_width , 0);
	HardwareInstr * mem_instr = push_hw_instruction(prog , COMP_MEMORY , 3 , 1);
	
	mem_instr -> inputs[0] = first_addr;
	mem_instr -> inputs[1] = last_addr;
	mem_instr -> inputs[2] = PC;
	mem_instr -> outputs[0] = mem_out_addr;
	
	push_flip_flop(prog , IR , mem_out_addr , clock , chip_select);
}



CircuitPort * push_port(CircuitProgram * prog)
{
	CircuitPort * result = 0;
	CircuitPort * ports = prog -> ports;
	u32 count = prog -> port_count;
	u32 capacity = prog -> port_capacity;
	
	assert(count < capacity);
	result = ports + count++;
	*result = {};
	
	result -> id = count;
	prog -> port_count = count;
	
	return(result);
}


void MIPS_generate_ports(
						 CircuitProgram * prog , Mem line_arena , Mem port_spec_arena)
{
	
	PortSpec * port_specs = mem_start(port_spec_arena , PortSpec);
	u32 port_count = mem_size(port_spec_arena , PortSpec);
	
	for(u32 port_index = 0; port_index < port_count; port_index++)
	{
		
		PortSpec * port_spec = port_specs + port_index;
		CircuitPort * port = push_port(prog);
		CircuitAddress addr = get_line_addr(line_arena , port_spec -> line_id);
		
		port -> id = port_spec -> line_id;
		port -> addr = addr;
		port -> direction = port_spec -> direction;
		port -> min = port_spec -> min;
		port -> max = port_spec -> max;
		port -> sign = port_spec -> sign;
		
		str_copy(port -> filename , port_spec -> filename);
	}
}

void MIPS_generate_clock_logic(
							   CircuitProgram * prog , CircuitAddress clock , CircuitAddress ss_addr_space)
{
	
	HardwareInstr * instr = push_hw_instruction(prog , COMP_CIRCUIT_STEADY_STATE , 1 , 1);
	
	instr -> inputs[0] = ss_addr_space;
	instr -> outputs[0] = clock;
	
}


void MIPS_make_IR_field_connections(CircuitProgram * prog , Mem fields_arena , Mem line_arena , CircuitAddress IR)
{
	
	InstrFieldSpec * fields = mem_start(fields_arena , InstrFieldSpec);
	u32 field_count = mem_size(fields_arena , InstrFieldSpec);
	
	for(u32 field_index = 0; field_index < field_count; field_index++)
	{
		InstrFieldSpec * curr_field = fields + field_index;
		u32 min = curr_field -> min;
		u32 max = curr_field -> max;
		u32 line_id = curr_field -> line_id;
		
		CircuitAddress f_addr = get_line_addr(line_arena , line_id);
		CircuitAddress s_addr = sub_addr(IR , min , max);
		connect(prog , s_addr , f_addr);
	}
}


struct InstrStageLine
{
	CircuitAddress addr;
	u32 id;
};

struct InstrStage
{
	
	u32 id;
	u32 start_event_index;
	u32 ss_event_index;
	u32 end_event_index;
	u32 state;
	i32 last_timestamp;
	
	i32 time;
	i32 period;
	u32 pc_value;
};

struct CompletedInstr
{
	u32 start_stage_index;
	u32 end_stage_index;
	u32 pc_value;
	i32 time;
};

struct InstrTracker
{
	InstrStageLine * stage_lines;
	u32 stage_line_capacity;
	u32 stage_line_count;
	
	InstrStage * stage_list;
	u32 stage_count;
	u32 stage_capacity;
	
	CompletedInstr * completed_instrs;
	u32 completed_instr_count;
	u32 completed_instr_capacity;
	
	CircuitAddress pc_addr;
};


struct CircuitAddressPower
{
	i32 time;
	CircuitAddress addr;
	
	u32 before_value;
	u32 after_value;
	
	u32 switches;
	u32 fanout;
};

struct CircuitModulePower
{
	u32 time;
	u32 module_id;
	
	u32 norm_static_energy;
	u32 norm_switch_energy;
	u32 switches;
	u32 fanout;
};


struct StatFrame
{
	u32 time;
	u32 norm_static_energy;
	u32 norm_switch_energy;
	u32 norm_short_circuit_energy;
	u32 fanout;
	u32 switches;
};

struct StatTracker
{
	
	u32 unit_static_energy;
	u32 unit_gate_capacitance;
	
	// NOTE: retained data
	
	u32 frames_recorded;
	u32 stages_recorded;
	u32 instrs_recorded;
	
	u32 total_energy;
	u32 total_switches;
	u32 total_stage_period;
	
	u32 max_energy;
	u32 max_switches;
	u32 max_stage_period;
	
	i32 last_frame_time;
	i32 last_stage_time;
	i32 last_instr_time;
	
	// NOTE: transient data
	
	StatFrame * frames;
	u32 frame_count;
	u32 frame_capacity;
	
	Mem addr_power_arena;
	Mem module_power_arena;
	
};




void do_MIPS_Rtype_instr_execute_tests(
									   Instr sw_instr , Mem line_arena , Mem reg_arena , u32 reg_width , 
									   i32 start_time , i32 end_time , i32 time , EventBuffer * event_buffer , 
									   CircuitState * curr_state , Mem * arena)
{
	
	u32 op_code = sw_instr.op_code;
	u32 rs = sw_instr.rs;
	u32 rt = sw_instr.rt;
	
	RegisterSpec * input_reg_1 = get_register_spec(reg_arena , rs);
	RegisterSpec * input_reg_2 = get_register_spec(reg_arena , rt);
	
	CircuitAddress input_reg_addr_1 = get_line_addr(line_arena , input_reg_1 -> line_id);
	CircuitAddress input_reg_addr_2 = get_line_addr(line_arena , input_reg_2 -> line_id);
	
	CircuitAddress input_data_addr_1 = get_line_addr(line_arena , MIPS_REG_INPUT_DATA_1_LINE);
	CircuitAddress input_data_addr_2 = get_line_addr(line_arena , MIPS_REG_INPUT_DATA_2_LINE);
	CircuitAddress output_data_1_addr = get_line_addr(line_arena , MIPS_REG_OUTPUT_DATA_1_LINE);
	
	CircuitAddress mult_out_addr = get_line_addr(line_arena , MIPS_MULT_OUT_LINE);
	CircuitAddress adder_out_addr = get_line_addr(line_arena , MIPS_ADDER_OUT_LINE);
	
	u32 signal_result_1 = signal_trace_exists(input_reg_addr_1 , input_data_addr_1 , end_time , event_buffer , arena);
	u32 signal_result_2 = signal_trace_exists(input_reg_addr_2 , input_data_addr_2 , end_time , event_buffer , arena);
	
	i32 lhs = get_cv_to_int_at(input_reg_addr_1 , 1 , start_time , time , event_buffer , curr_state);
	i32 rhs = get_cv_to_int_at(input_reg_addr_2 , 1 , start_time , time , event_buffer , curr_state);
	i32 result = get_cv_to_int_at(output_data_1_addr , 1 ,  end_time , time , event_buffer , curr_state);
	
	assert(signal_result_1);
	assert(signal_result_2);
	
	switch(op_code)
	{
		
		case(OP_HALT):
		{
			break;
		}
		
		case(OP_MULT):
		{
			i32 expected_result = (lhs*rhs)%MAX_SIGNED_VALUE(reg_width);
			u32 signal_result = signal_trace_exists(mult_out_addr , output_data_1_addr , end_time , event_buffer , arena);
			
			assert(signal_result);
			assert(result == expected_result);
			break;
		}
		
		case(OP_ADD):
		{
			
			i32 expected_result = (lhs+rhs)%MAX_SIGNED_VALUE(reg_width);
			u32 signal_result = signal_trace_exists(adder_out_addr , output_data_1_addr , end_time , event_buffer , arena);
			
			assert(signal_result);
			assert(result == expected_result);
			break;
		}
		
		case(OP_LOAD):
		case(OP_STORE):
		case(OP_JUMP):
		{
			
			break;
		}
		
		default:
		{
			assert_zero;
		}
	}
}

void do_MIPS_Itype_instr_execute_tests(
									   Instr sw_instr , Mem line_arena , Mem reg_arena , u32 reg_width , 
									   i32 start_time , i32 end_time , i32 time , EventBuffer * event_buffer , 
									   CircuitState * curr_state , Mem * arena)
{
	
	
	u32 op_code = sw_instr.op_code;
	u32 rs = sw_instr.rs;
	u32 rt = sw_instr.rt;
	
	RegisterSpec * input_reg_1 = get_register_spec(reg_arena , rs);
	RegisterSpec * input_reg_2 = get_register_spec(reg_arena , rt);
	
	CircuitAddress input_reg_addr_1 = get_line_addr(line_arena , input_reg_1 -> line_id);
	CircuitAddress input_reg_addr_2 = get_line_addr(line_arena , input_reg_2 -> line_id);
	
	CircuitAddress input_data_addr_1 = get_line_addr(line_arena , MIPS_REG_INPUT_DATA_1_LINE);
	CircuitAddress input_data_addr_2 = get_line_addr(line_arena , MIPS_REG_INPUT_DATA_2_LINE);
	CircuitAddress output_data_addr_1 = get_line_addr(line_arena , MIPS_REG_OUTPUT_DATA_1_LINE);
	
	CircuitAddress adder_out_addr = get_line_addr(line_arena , MIPS_ADDER_OUT_LINE);
	CircuitAddress imm_addr = get_line_addr(line_arena , IR_IMM_LINE);
	
	i32 input_value_1 = get_cv_to_int_at(input_reg_addr_1 , 1 , start_time , time , event_buffer , curr_state);
	//i32 input_value_2 = get_cv_to_int_at(input_reg_addr_2 , 1 , start_time , time , event_buffer , curr_state);
	i32 output_value_1 = get_cv_to_int_at(output_data_addr_1 , 1 ,  end_time , time , event_buffer , curr_state);
	i32 imm_value = get_cv_to_int_at(imm_addr , 1 ,  start_time , time , event_buffer , curr_state);
	
	u32 signal_result_1 = signal_trace_exists(input_reg_addr_1 , input_data_addr_1 , end_time , event_buffer , arena);
	u32 signal_result_2 = signal_trace_exists(input_reg_addr_2 , input_data_addr_2 , end_time , event_buffer , arena);
	
	assert(signal_result_1);
	assert(signal_result_2);
	
	switch(op_code)
	{
		
		case(OP_HALT):
		{
			break;
		}
		
		case(OP_ADD_I):
		{
			
			i32 expected_result = (input_value_1+imm_value)%MAX_SIGNED_VALUE(reg_width);
			u32 signal_result = signal_trace_exists(adder_out_addr , output_data_addr_1 , end_time , event_buffer , arena);
			
			assert(signal_result);
			assert(output_value_1 == expected_result);
			break;
		}
		
		
		case(OP_SHIFT_LEFT):
		{
			
			u32 bit_mask = get_bit_bound_mask(0 , reg_width);
			i32 expected_result = (input_value_1 << imm_value) & bit_mask;
			u32 signal_result = signal_trace_exists(adder_out_addr , output_data_addr_1 , end_time , event_buffer , arena);
			
			assert(signal_result);
			assert(output_value_1 == expected_result);
			break;
		}
		
		
		case(OP_SHIFT_RIGHT):
		{
			
			u32 bit_mask = get_bit_bound_mask(0 , reg_width);
			i32 expected_result = (input_value_1 >> imm_value) & bit_mask;
			u32 signal_result = signal_trace_exists(adder_out_addr , output_data_addr_1 , end_time , event_buffer , arena);
			
			assert(signal_result);
			assert(output_value_1 == expected_result);
			break;
		}
		
		case(OP_LOAD):
		{
			
			
			break;
		}
		
		case(OP_STORE):
		case(OP_JUMP):
		{
			
			break;
		}
		
		default:
		{
			assert_zero;
		}
	}
}


void do_MIPS_Jtype_instr_execute_tests(
									   Instr sw_instr , Mem line_arena ,
									   i32 exe_start_time , i32 exe_end_time , i32 time , EventBuffer * event_buffer , 
									   CircuitState * curr_state , Mem * arena)
{
	
	CircuitAddress PC_addr = get_line_addr(line_arena , PC_LINE);
	
	switch(sw_instr.op_code)
	{
		case(OP_JUMP):
		{
			
			i32 pc_before = get_cv_to_int_at(
											 PC_addr ,  0 , 
											 exe_start_time , time , 
											 event_buffer , curr_state
											 );
			
			i32 pc_after = get_cv_to_int_at(
											PC_addr , 0 , 
											exe_end_time , time , 
											event_buffer , curr_state
											);
			
			i32 jump_value = sw_instr.pc_rel_jump;
			i32 expected_pc = pc_before + jump_value;
			
			assert(pc_before >= jump_value);
			assert(pc_after == expected_pc);
			
			break;
		}
	}
}

void do_MIPS_Rtype_instr_write_back_tests(
										  Instr sw_instr , Mem line_arena , Mem reg_arena , u32 reg_width , 
										  i32 target_time , i32 time , EventBuffer * event_buffer , CircuitState * curr_state , Mem * arena)
{
	
	RegisterSpec * rd_reg = get_register_spec(reg_arena , sw_instr.rd);
	CircuitAddress reg_addr = get_line_addr(line_arena , rd_reg -> line_id);
	CircuitAddress reg_output_data_addr = get_line_addr(line_arena , MIPS_REG_OUTPUT_DATA_1_LINE);
	
	if(rd_reg -> type == GP_REGISTER)
	{
		u32 trace_result = signal_trace_exists(reg_output_data_addr , reg_addr , target_time , event_buffer , arena);
		assert(trace_result);
	}
}

void do_MIPS_Itype_instr_write_back_tests(
										  Instr sw_instr , Mem line_arena , Mem reg_arena , u32 reg_width , 
										  i32 target_time , i32 time , EventBuffer * event_buffer , CircuitState * curr_state , Mem * arena)
{
	
	RegisterSpec * rt_reg = get_register_spec(reg_arena , sw_instr.rt);
	CircuitAddress reg_addr = get_line_addr(line_arena , rt_reg -> line_id);
	CircuitAddress reg_output_data_addr = get_line_addr(line_arena , MIPS_REG_OUTPUT_DATA_1_LINE);
	
	if(sw_instr.op_code == OP_ADD_I)
	{
		if(rt_reg -> type == GP_REGISTER)
		{
			u32 trace_result = signal_trace_exists(reg_output_data_addr , reg_addr , target_time , event_buffer , arena);
			assert(trace_result);
		}
	}
}

void do_MIPS_control_lines_tests(
								 Instr sw_instr , Mem line_arena , Mem instr_arena , u32 reg_width , 
								 i32 target_time , i32 time , 
								 EventBuffer * event_buffer , CircuitState * curr_state)
{
	
	InstrSpec * instrs = mem_start(instr_arena , InstrSpec);
	u32 instr_count = mem_size(instr_arena , InstrSpec);
	
	for(u32 instr_index = 0; instr_index < instr_count; instr_index++)
	{
		
		InstrSpec * instr = instrs + instr_index;
		u32 * line_ids = instr -> ctr_line_ids;
		u32 * line_values = instr -> ctr_line_values;
		u32 line_count = instr -> ctr_line_count;
		u32 op_code = instr -> op_code;
		
		if(op_code == sw_instr.op_code)
		{
			for(u32 line_index = 0; line_index < line_count; line_index++)
			{
				
				LineSpec * line = get_line_spec(line_arena , line_ids[line_index]);
				CircuitAddress line_addr = line -> addr;
				u32 expected_value = line_values[line_index];
				u32 value =  get_cv_to_int_at(
											  line_addr , 0 , 
											  target_time , time , 
											  event_buffer , curr_state
											  );
				
				assert(value == expected_value);
			}	
		}
	}
}




void test_IR_integrity(
					   Instr instr , u32 instr_type , Mem line_arena , Mem field_arena , 
					   i32 target_time , i32 time , 
					   EventBuffer * event_buffer , CircuitState * curr_state)
{
	
	InstrFieldSpec fields[16] = {};
	u32 field_count = get_instr_fields(field_arena , instr_type , fields);
	
	for(u32 field_index = 0; field_index < field_count; field_index++)
	{
		
		InstrFieldSpec field = fields[field_index];
		CircuitAddress addr = get_line_addr(line_arena,  field.line_id);
		
		u32 value = get_field_value(instr , field);
		CircuitValue expected_cv = field_to_cv(value , field);
		CircuitValue cv = get_cv_at(
									addr , 
									target_time , time , 
									event_buffer , curr_state
									);
		
		assert(cv.v == expected_cv.v && cv.width == expected_cv.width);
	}
}

void set_instr_tracker(InstrTracker * tracker , Mem * arena , Mem stage_line_arena , Mem line_arena)
{
	
	
	StageLineSpec * stage_line_specs = mem_start(stage_line_arena , StageLineSpec);
	u32 line_count = mem_size(stage_line_arena , StageLineSpec);
	assert(array_size(tracker -> stage_lines) <= line_count);
	
	tracker -> stage_line_count = line_count;
	tracker -> stage_line_capacity = 16;
	tracker -> stage_lines = mem_push_array(arena , InstrStageLine , tracker -> stage_line_capacity); 
	assert(line_count <= tracker -> stage_line_capacity);
	
	tracker -> stage_capacity = 64;
	tracker -> stage_list = mem_push_array(arena , InstrStage , tracker -> stage_capacity);
	
	tracker -> completed_instr_capacity = 32;
	tracker -> completed_instrs = mem_push_array(arena , CompletedInstr , tracker -> completed_instr_capacity);
	
	for(u32 stage_index = 0; stage_index < line_count; stage_index++)
	{
		
		u32 line_id = stage_line_specs[stage_index].line_id;
		u32 stage_id = stage_line_specs[stage_index].stage_id;
		
		tracker -> stage_lines[stage_index].addr = get_line_addr(line_arena , line_id);
		tracker -> stage_lines[stage_index].id = stage_id;
	}
	
	tracker -> pc_addr = get_line_addr(line_arena , PC_LINE);
	
}


inline
InstrStageLine * get_instr_stage_line(InstrTracker * tracker , u32 stage_id)
{
	
	InstrStageLine * result = 0;
	InstrStageLine * stage_lines = tracker -> stage_lines;
	u32 line_count = tracker -> stage_line_count;
	
	
	for(u32 line_index = 0; line_index < line_count; line_index++)
	{
		
		InstrStageLine * stage = stage_lines + line_index;
		if(stage -> id == stage_id)
		{
			result = stage;
			break;
		}
	}
	
	return(result);
}


inline
InstrStage * get_stage(InstrTracker * tracker , CompletedInstr * completed_instr , u32 stage_id)
{
	
	InstrStage * result = 0;
	InstrStage * stage_list = tracker -> stage_list;
	u32 start_stage_index = completed_instr -> start_stage_index;	
	u32 end_stage_index = completed_instr -> end_stage_index;	
	
	for(u32 stage_index = start_stage_index; stage_index < end_stage_index; stage_index++)
	{
		
		InstrStage * stage = stage_list + stage_index;
		if(stage -> id == stage_id)
		{
			result = stage;
			break;
		}
	}
	
	return(result);
}


inline
void perform_MIPS_path_tests(
							 Instr sw_instr , 
							 Mem line_arena , Mem reg_spec_arena , 
							 Mem instr_spec_arena , Mem field_spec_arena , u32 reg_width , 
							 CompletedInstr * completed_instr , InstrSpec * instr_spec , 
							 InstrTracker * tracker , EventBuffer * event_buffer , 
							 CircuitState * curr_state , i32 time , Mem * arena)
{
	
	CircuitEvent * events = event_buffer -> events;
	
	u32 instr_type = instr_spec -> type;
	InstrStage decode = *get_stage(tracker , completed_instr , INSTR_STAGE_DECODE);
	InstrStage execute = *get_stage(tracker , completed_instr , INSTR_STAGE_EXECUTE);
	
	CircuitEvent decode_end = events[decode.end_event_index];
	CircuitEvent execute_start = events[execute.start_event_index];
	CircuitEvent execute_end = events[execute.end_event_index];
	
	switch(instr_type)
	{
		case(INSTR_R_TYPE):
		{
			
			do_MIPS_Rtype_instr_execute_tests(
											  sw_instr , line_arena , reg_spec_arena , reg_width , 
											  execute_start.time , execute_end.time , time , event_buffer , 
											  curr_state , arena
											  );
			
			do_MIPS_Rtype_instr_write_back_tests(
												 sw_instr , line_arena , reg_spec_arena , reg_width , 
												 execute_end.time , time , event_buffer , curr_state , arena
												 );
			
			break;
		}
		
		case(INSTR_I_TYPE):
		{
			
			do_MIPS_Itype_instr_execute_tests(
											  sw_instr , line_arena , reg_spec_arena , reg_width , 
											  execute_start.time , execute_end.time , time , event_buffer , 
											  curr_state , arena
											  );
			
			do_MIPS_Itype_instr_write_back_tests(
												 sw_instr , line_arena , reg_spec_arena , reg_width , 
												 execute_end.time , time , event_buffer , curr_state , arena
												 );
			
			break;
			
		}
		
		case(INSTR_J_TYPE):
		{
			
			do_MIPS_Jtype_instr_execute_tests(
											  sw_instr , line_arena , 
											  execute_start.time , execute_end.time , time , event_buffer , 
											  curr_state , arena
											  );
			
			
			break;
		}
	}
	
	do_MIPS_control_lines_tests(
								sw_instr , line_arena , instr_spec_arena , reg_width , 
								decode_end.time , time , event_buffer , curr_state
								);
	
}


void perform_DOT4_path_tests(
							 Mem line_arena , Mem reg_arena , 
							 CompletedInstr * completed_instr , InstrSpec * instr_spec , 
							 InstrTracker * tracker , EventBuffer * event_buffer , 
							 CircuitState * curr_state , i32 time , Mem * arena)
{
	
	InstrStage exe = *get_stage(tracker , completed_instr , INSTR_STAGE_EXECUTE);
	
	CircuitEvent * events = event_buffer -> events;
	CircuitEvent exe_end = events[exe.end_event_index];
	
	CircuitAddress output_addr = get_line_addr(line_arena , REG_WRITE_DATA_LINE);
	
	CircuitAddress A1_addr = get_reg_addr(reg_arena , line_arena , "r1");
	CircuitAddress A2_addr = get_reg_addr(reg_arena , line_arena , "r2");
	CircuitAddress A3_addr = get_reg_addr(reg_arena , line_arena , "r3");
	CircuitAddress A4_addr = get_reg_addr(reg_arena , line_arena , "r4");
	
	CircuitAddress B1_addr = get_reg_addr(reg_arena , line_arena , "r5");
	CircuitAddress B2_addr = get_reg_addr(reg_arena , line_arena , "r6");
	CircuitAddress B3_addr = get_reg_addr(reg_arena , line_arena , "r7");
	CircuitAddress B4_addr = get_reg_addr(reg_arena , line_arena , "r8");
	
	i32 A1 = get_cv_to_int_at(A1_addr , 1 , exe_end.time , time , event_buffer , curr_state);
	i32 A2 = get_cv_to_int_at(A2_addr , 1 , exe_end.time , time , event_buffer , curr_state);
	i32 A3 = get_cv_to_int_at(A3_addr , 1 , exe_end.time , time , event_buffer , curr_state);
	i32 A4 = get_cv_to_int_at(A4_addr , 1 , exe_end.time , time , event_buffer , curr_state);
	
	i32 B1 = get_cv_to_int_at(B1_addr , 1 , exe_end.time , time , event_buffer , curr_state);
	i32 B2 = get_cv_to_int_at(B2_addr , 1 , exe_end.time , time , event_buffer , curr_state);
	i32 B3 = get_cv_to_int_at(B3_addr , 1 , exe_end.time , time , event_buffer , curr_state);
	i32 B4 = get_cv_to_int_at(B4_addr , 1 , exe_end.time , time , event_buffer , curr_state);
	
	i32 output = get_cv_to_int_at(output_addr , 1 , exe_end.time , time , event_buffer , curr_state);
	i32 dot_value = A1*B1 + A2*B2 + A3*B3 + A4*B4;
	
	assert(output == dot_value);
	
}


void test_only_data_path_enabled(
								 DataPathSpec * path , Mem path_arena , Mem line_arena , 
								 EventBuffer * event_buffer , CircuitState * curr_state , i32 target_time , i32 time , Mem * arena)
{
	
	
	DataPathSpec * paths = mem_start(path_arena , DataPathSpec);
	u32 path_count = mem_size(path_arena , DataPathSpec);
	
	for(u32 path_index = 0; path_index < path_count; path_index++)
	{
		
		DataPathSpec * curr_path = paths + path_index;
		CircuitAddress enable = get_line_addr(line_arena , curr_path -> enable_line);
		u32 val = get_cv_to_int_at(
								   enable , 0 , 
								   target_time , time , 
								   event_buffer , curr_state
								   );
		
		assert(width_of(enable) == 1);
		if(path -> id == curr_path -> id) 
		{
			assert(val == 1);
		}
		else
		{
			assert(val == 0);
		}
	}
}

void test_instr_stages(
					   InstrTracker * tracker , 
					   Mem line_arena , Mem reg_spec_arena , 
					   Mem instr_spec_arena , Mem field_spec_arena , Mem path_arena ,
					   u32 reg_width , Instr * sw_instrs , u32 sw_instr_count , 
					   EventBuffer * event_buffer , CircuitState * curr_state , i32 time , Mem * arena)
{
	
	
	CompletedInstr * completed_instrs = tracker -> completed_instrs;
	u32 completed_instr_count = tracker -> completed_instr_count;
	
	CircuitAddress PC_addr = get_line_addr(line_arena , PC_LINE);
	CircuitEvent * events = event_buffer -> events;
	
	for(
		u32 completed_index = 0; 
		completed_index < completed_instr_count; 
		completed_index++)
	{
		
		CompletedInstr * completed_instr = completed_instrs + completed_index;
		InstrStage fetch = *get_stage(tracker , completed_instr , INSTR_STAGE_FETCH);
		InstrStage decode = *get_stage(tracker , completed_instr , INSTR_STAGE_DECODE);
		InstrStage execute = *get_stage(tracker , completed_instr , INSTR_STAGE_EXECUTE);
		
		CircuitEvent fetch_ss = events[fetch.ss_event_index];
		CircuitEvent decode_ss = events[decode.ss_event_index];
		CircuitEvent execute_ss = events[execute.ss_event_index];
		
		u32 program_counter = get_cv_to_int_at(PC_addr , 0 , fetch_ss.time , time , event_buffer , curr_state);
		assert(program_counter <= sw_instr_count);
		
		Instr sw_instr = sw_instrs[program_counter];
		
		InstrSpec * instr_spec = get_instr_spec(instr_spec_arena , sw_instr.op_code);
		u32 instr_type = instr_spec -> type;
		
		DataPathSpec * path = get_data_path_from_op_code(path_arena , instr_spec_arena , sw_instr.op_code);
		u32 path_id = path -> id;
		
		test_only_data_path_enabled(path , path_arena , line_arena , event_buffer , curr_state , execute_ss.time , time , arena);
		
		test_IR_integrity(
						  sw_instr , instr_type , line_arena , field_spec_arena , 
						  decode_ss.time , time , event_buffer , 
						  curr_state
						  );
		
		switch(path_id)
		{
			case(MIPS_PATH):
			{
				
				perform_MIPS_path_tests(
										sw_instr , line_arena , reg_spec_arena , 
										instr_spec_arena , field_spec_arena , reg_width , 
										completed_instr , instr_spec , 
										tracker , event_buffer , 
										curr_state , time , arena
										);
				
				break;
			}
			
			case(DOT4_PATH):
			{
				perform_DOT4_path_tests(
										line_arena , reg_spec_arena ,
										completed_instr , instr_spec , 
										tracker , event_buffer , 
										curr_state , time , arena
										);
				
				break;
			}
		}
		
		completed_instr -> pc_value = program_counter;
	}
}

void clear_stage_list(InstrStage * stage_list , u32 count)
{
	
	for(u32 stage_index = 0; stage_index < count; stage_index++)
	{
		stage_list[stage_index].start_event_index = 0;
		//stage_list[stage_index].ss_event_index = 0;
		stage_list[stage_index].end_event_index = 0;
		stage_list[stage_index].state = 0;
	}
}

u32 get_last_event_index(EventBuffer * event_buffer)
{
	
	u32 start = event_buffer -> event_start_index;
	u32 end = event_buffer -> event_end_index;
	u32 index = end;
	
	if(start != end)
	{
		if(!index) 	index = event_buffer -> event_capacity;
		else 		index--;
	}
	
	return(index);
}



void get_instr_stages(
					  InstrTracker * tracker ,
					  EventBuffer * event_buffer)
{
	
	
	InstrStageLine * stage_lines = tracker -> stage_lines;
	u32 stage_line_count = tracker -> stage_line_count;
	u32 stages_per_instr = tracker -> stage_line_count;
	
	InstrStage * stage_list = tracker -> stage_list;
	u32 stage_count = 0;
	u32 stage_capacity = tracker -> stage_capacity;
	
	CompletedInstr * completed_instrs = tracker -> completed_instrs;
	u32 completed_instr_count = 0;
	u32 completed_instr_capacity = tracker -> completed_instr_capacity;
	
	CircuitEvent * events = event_buffer -> events;
	u32 capacity = event_buffer -> event_capacity;
	u32 start_index = event_buffer -> event_start_index;
	u32 end_index = event_buffer -> event_end_index;
	u32 curr_event_index = start_index;
	
	mem_set(stage_list , 0 , sizeof(InstrStage)*stage_capacity);
	mem_set(completed_instrs , 0 , sizeof(CompletedInstr)*completed_instr_count);
	
	
	u32 pc_values[128] = {};
	i32 pc_times[128] = {};
	u32 pc_count = 0;
	u32 pc_capacity = array_size(pc_values);
	i32 last_pc_time = -1;
	CircuitAddress pc_addr = tracker -> pc_addr;
	
	while(curr_event_index != end_index)
	{
		
		CircuitEvent * event = events + curr_event_index;
		
		if(event -> dest == pc_addr && event -> type == CIRCUIT_SIGNAL_EVENT)
		{
			
			if(last_pc_time < event -> time)
			{
				assert(pc_count < pc_capacity);
				pc_values[pc_count] = cv_to_int(event -> result_value , 0);
				pc_times[pc_count] = event -> time;
				pc_count++;
				last_pc_time = event -> time;
			}
		}
		
		else if(event -> type == CIRCUIT_SIGNAL_EVENT)
		{
			for(u32 line_index = 0; line_index < stage_line_count; line_index++)
			{
				
				InstrStageLine * stage_line = stage_lines + line_index;
				CircuitAddress line_addr = stage_line -> addr;
				u32 stage_id = stage_line -> id;
				
				if(line_addr == event -> dest)
				{
					
					CircuitValue before_value = event -> original_value;
					CircuitValue after_value = event -> result_value;
					u32 pos_edge = !before_value.v && after_value.v;
					u32 neg_edge = before_value.v && !after_value.v;
					InstrStage * target_stage = 0;
					
					if(pos_edge || neg_edge)
					{
						
						assert(pos_edge ^ neg_edge);
						for(u32 stage_index = 0; stage_index < stage_count; stage_index++)
						{
							InstrStage * stage = stage_list + stage_index;
							
							if( stage -> id == stage_id && 
							   (stage -> last_timestamp == event -> time || stage -> state < 2))
							{
								target_stage = stage;
								break;
							}
						}
						
						if(!target_stage && pos_edge)
						{
							assert(stage_count < stage_capacity);
							target_stage = stage_list + stage_count++;
							target_stage -> id = stage_id;
							target_stage -> time = event -> time;
							target_stage -> period -= event -> time;
						}
						
						if(target_stage)
						{
							if(pos_edge && target_stage -> state == 0)
							{
								target_stage -> start_event_index = curr_event_index;
								target_stage -> state++;
							}
							
							else if(neg_edge && target_stage -> state == 1)
							{
								target_stage -> end_event_index = curr_event_index;
								target_stage -> period += event -> time;
								target_stage -> state++;
							}
							
							else
							{
								assert(target_stage -> last_timestamp == event -> time);
							}
							
							target_stage -> last_timestamp = event -> time;
						}
					}
				}
			}
		}
		
		if(event -> type == STEADY_STATE_EVENT)
		{
			
			for(u32 stage_index = 0; stage_index < stage_count; stage_index++)
			{
				
				InstrStage * stage = stage_list + stage_index;
				if(stage -> state == 1)
				{
					if(stage -> last_timestamp <= event -> time)
					{
						stage -> ss_event_index = curr_event_index;
					}
				}
			}
		}
		
		if(++curr_event_index >= capacity) curr_event_index = 0;
	}
	
	for(u32 stage_index = 0; stage_index < stage_count; stage_index++)
	{
		
		u32 found = 0;
		InstrStage * stage = stage_list + stage_index;
		
		for(u32 pc_index = 0; pc_index < pc_count; pc_index++)
		{
			if(pc_times[pc_index] == stage -> time)
			{
				stage -> pc_value = pc_values[pc_index];
				found = 1;
				break;
			}
		}
		
		assert(found);
	}
	
	for(u32 stage_index = 0; stage_index+stages_per_instr < stage_count; stage_index++)
	{
		
		InstrStage fetch = stage_list[stage_index+0];
		InstrStage decode = stage_list[stage_index+1];
		InstrStage execute = stage_list[stage_index+2];
		
		if(fetch.id == INSTR_STAGE_FETCH)
		{
			
			assert(decode.id == INSTR_STAGE_DECODE);
			assert(execute.id == INSTR_STAGE_EXECUTE);
			
			if(fetch.state == 2 && decode.state == 2 && execute.state == 2)
			{
				
				assert(completed_instr_count < completed_instr_capacity);
				CompletedInstr * comp = completed_instrs + completed_instr_count++;
				comp -> start_stage_index = stage_index;
				comp -> end_stage_index = stage_index+stages_per_instr;
				comp -> pc_value = fetch.pc_value;
				comp -> time = fetch.time;
			}
		}
	}
	
	tracker -> stage_count = stage_count;
	tracker -> completed_instr_count = completed_instr_count;
}


u32 end_of_sw_instr(
					InstrTracker * tracker , CircuitState * prev_state , CircuitState * curr_state)
{
	
	InstrStageLine * stage = get_instr_stage_line(tracker , INSTR_STAGE_FETCH);
	u32 result = is_pos_edge(
							 stage -> addr , 
							 prev_state , curr_state
							 );
	return(result);
}


void check_fields(Mem field_arena , u32 instr_type)
{
	
	InstrFieldSpec fields[32] = {};
	u32 field_count = get_instr_fields(field_arena , instr_type , fields);
	u32 mask = 0;
	
	for(u32 field_index = 0; field_index < field_count; field_index++)
	{
		InstrFieldSpec field = fields[field_index];
		u32 field_mask = get_bit_bound_mask(field.min , field.max);
		
		assert(!(mask & field_mask));
		mask |= field_mask;
	}
}

void do_ctr_spec_checks(Mem line_arena , ControlUnitSpec * ctr_spec)
{
	
	u32 * ctr_lines = ctr_spec -> ctr_lines;
	u32 ctr_line_count = ctr_spec -> ctr_line_count;
	
	for(u32 ctr_line_index = 0; ctr_line_index < ctr_line_count; ctr_line_index++)
	{
		LineSpec * line = get_line_spec(line_arena , ctr_lines[ctr_line_index]);
		assert(line);
	}
}

void do_spec_checks(MipsSpec * spec)
{
	
	
	Mem line_arena = spec -> lines;
	Mem field_arena = spec -> fields;
	
	StageLineSpec * stage_lines = mem_start(spec -> stage_lines , StageLineSpec);
	u32 stage_count = mem_size(spec -> stage_lines , StageLineSpec);
	
	InstrSpec * instrs = mem_start(spec -> instr_set , InstrSpec);
	u32 instr_count = mem_size(spec -> instr_set , InstrSpec);
	
	for(u32 stage_index = 0; stage_index < stage_count; stage_index++)
	{
		LineSpec * line = get_line_spec(line_arena , stage_lines[stage_index].line_id);
		assert(line);
	}
	
	
	for(u32 instr_index = 0; instr_index < instr_count; instr_index++)
	{
		
		InstrSpec * instr = instrs + instr_index;
		u32 * line_ids = instr -> ctr_line_ids;
		u32 line_count = instr -> ctr_line_count;
		
		for(u32 line_index = 0; line_index < line_count; line_index++)
		{
			
			u32 id = line_ids[line_index];
			LineSpec * line = get_line_spec(line_arena , id);
			
			assert(line);
		}
	}
	
	// NOTE: check for overlap in instruction fieds.
	check_fields(field_arena , INSTR_R_TYPE);
	check_fields(field_arena , INSTR_I_TYPE);
	check_fields(field_arena , INSTR_J_TYPE);
	
}



void MIPS_generate_data_mem(
							CircuitProgram * prog , CircuitPort * ports , u32 port_count , 
							CircuitAddress clock , CircuitAddress chip_select , 
							CircuitAddress data_out , CircuitAddress addr_in , CircuitAddress data_in ,
							CircuitAddress read_select , CircuitAddress write_select)
{
	
	CircuitPort * input_port = ports + 0;
	CircuitPort * output_port = ports + 1;
	
	CircuitAddress input_port_addr = input_port -> addr;
	CircuitAddress output_port_addr = output_port -> addr;
	
	CircuitAddress read = push_addr(prog , 1);
	CircuitAddress write = push_addr(prog , 1);
	
	push_and2(prog , read , chip_select , read_select);
	push_and2(prog , write , chip_select , write_select);
	
	connect(prog , data_in , output_port_addr);
	connect(prog , input_port_addr , data_out);
	
	//push_flip_flop(prog , read_out , input_port_addr , clock , read);
	//push_flip_flop(prog , output_port_addr , write_in , clock , write);
	
	input_port -> clock = clock;
	input_port -> enable = read;
	
	output_port -> clock = clock;
	output_port -> enable = write;
	
}




// NOTE: "space data path" means a data path where no two computations contend with the same data, if they were executed
// in parallel. The "space" implies that state that has been operated on by multiple operations in sequential order, have been split 
// into coppies, each of which have been assigned to one of the operations. This is transforming time into space. 

// NOTE: time is implicit with the DataPathPair's. Each pair implys a point in time, and therefore, the memory associated with that pair
// is unique in terms of time. However, this does not imply that a given memory unit is _physically_ unique for a given pair, and this
// because space is not implied in the model. Space is _explicitly_ defined as a register number. To translate from time to space, one must
// convert memory that has been used multiple times to other unused register-numbers.



struct DataPathPair
{
	
	Instr * instrA;
	Instr * instrB;
	
	u32 instrA_output_index;
	u32 instrB_input_index;
	u32 reg;
	u32 min_bit_addr;
	u32 max_bit_addr;
};


void get_data_path(
				   Instr * instructions , u32 instr_count , 
				   DataPathPair ** data_path_pairs_ptr , u32 * pair_count_ptr , 
				   Mem * arena)
{
	
	DataPathPair * data_pairs = mem_end(*arena , DataPathPair);
	u32 pair_capacity = mem_rem(*arena , DataPathPair);
	u32 pair_count = 0;
	
	// NOTE: ith element maps to register r<i>
	Instr * register_dependence[32] = {};
	u32 is_output[32] = {};
	
	for(u32 instr_index = 0; instr_index < instr_count; instr_index++)
	{
		
		assert(pair_count+1 < pair_capacity);
		
		Instr * curr_instr = instructions + instr_index;
		DataPathPair * pair1 = data_pairs + pair_count++;
		DataPathPair * pair2 = data_pairs + pair_count++;
		
		u32 input_reg1 = curr_instr -> rs;
		u32 input_reg2 = curr_instr -> rt;
		u32 output_reg = curr_instr -> rd;
		
		assert(input_reg1 < array_size(register_dependence));
		assert(input_reg2 < array_size(register_dependence));
		assert(output_reg < array_size(register_dependence));
		
		// TODO: acquire inputs depending on the instruction.
		
		pair1 -> instrA = register_dependence[input_reg1];
		pair1 -> instrB = curr_instr;
		pair1 -> instrA_output_index = 0;
		pair1 -> instrB_input_index = 0;
		pair1 -> reg = input_reg1;
		
		pair2 -> instrA = register_dependence[input_reg2];
		pair2 -> instrB = curr_instr;
		pair2 -> instrA_output_index = 0;
		pair2 -> instrB_input_index = 1;
		pair2 -> reg = input_reg2;
		
		register_dependence[output_reg] = curr_instr;
		is_output[input_reg1] = 0;
		is_output[input_reg2] = 0;
		is_output[output_reg] = 1;
	}
	
	
	// NOTE: finish off by adding the primary outputs.
	
	for(u32 reg_index = 0; reg_index < array_size(register_dependence); reg_index++)
	{
		
		Instr * instr = register_dependence[reg_index];
		if(is_output[reg_index])
		{
			
			assert(instr);
			assert(pair_count < pair_capacity);
			DataPathPair * pair = data_pairs + pair_count++;
			pair -> instrA = instr;
			pair -> instrB = 0;
			pair -> instrA_output_index = 0;
			pair -> instrB_input_index = 0; // NOTE: there is no instrB.
			pair -> reg = reg_index;
		}
	}
	
	mem_push_array(arena , DataPathPair , pair_count);
	*data_path_pairs_ptr = data_pairs;
	*pair_count_ptr = pair_count;
}

u32 generate_hardware_instructions(
								   HardwareInstr ** hw_instrs_ptr , u32 * hw_instr_count_ptr ,  
								   CircuitAddress ** primary_inputs_ptr , u32 * primary_input_count_ptr , 
								   CircuitAddress ** primary_outputs_ptr , u32 * primary_output_count_ptr , 
								   Instr * instructions , u32 instr_count , 
								   DataPathPair * data_path , u32 data_path_size , Mem * arena)
{
	
	HardwareInstr * hw_instrs = mem_end(*arena , HardwareInstr);
	u32 hw_instr_capacity = mem_rem(*arena , HardwareInstr);
	u32 hw_instr_count = 0;
	
	u32 curr_addr = 0;
	
	u32 pivot = 0;
	u32 * has_dep = mem_push_array(arena , u32 , data_path_size);
	
	while(pivot < data_path_size)
	{
		mem_set(has_dep , 0 , sizeof(u32)*data_path_size);
		for(u32 i = pivot; i < data_path_size; i++)
		{
			DataPathPair * curr_pair = data_path + i;
			for(u32 j = pivot; j < data_path_size; j++)
			{
				DataPathPair * target_pair = data_path + j;
				if(target_pair -> instrA && target_pair -> instrA == curr_pair -> instrB)
				{
					has_dep[j] = 1;
					assert(target_pair != curr_pair);
				}
			}
		}
		
		for(u32 i = pivot; i < data_path_size; i++)
		{
			DataPathPair * curr_pair = data_path + i;
			if(!has_dep[i])
			{
				DataPathPair temp = data_path[pivot];
				data_path[pivot] = *curr_pair;
				*curr_pair = temp;
				
				assert(pivot < data_path_size);
				pivot++;
			}
		}
	}
	
	mem_pop(arena , has_dep);
	
	for(u32 pair_index = 0; pair_index < data_path_size; pair_index++)
	{
		DataPathPair * pair = data_path + pair_index;
		pair -> min_bit_addr = curr_addr;
		pair -> max_bit_addr = curr_addr + 8;
		curr_addr += 8;
	}
	
	for(u32 instr_index = 0; instr_index < instr_count; instr_index++)
	{
		
		Instr * instr = instructions + instr_index;
		assert(hw_instr_count < hw_instr_capacity);
		HardwareInstr * hw_instr = hw_instrs + hw_instr_count++;
		u32 input_count = 0;
		u32 output_count = 0;
		
		for(u32 pair_index = 0; pair_index < data_path_size; pair_index++)
		{
			
			DataPathPair * pair = data_path + pair_index;
			if(pair -> instrB == instr)
			{
				assert(hw_instr -> inputs[pair -> instrB_input_index] == CIRCUIT_NULL_ADDR);
				hw_instr -> inputs[pair -> instrB_input_index] = CircuitAddress{pair -> min_bit_addr , pair -> max_bit_addr};
				input_count++;
			}
			
			if(pair -> instrA == instr)
			{
				assert(hw_instr -> outputs[pair -> instrA_output_index] == CIRCUIT_NULL_ADDR);
				hw_instr -> outputs[pair -> instrA_output_index] = CircuitAddress{pair -> min_bit_addr , pair -> max_bit_addr};
				output_count++;
			}
		}
		
		switch(instr -> op_code)
		{
			case(OP_ADD):
			{
				hw_instr -> function = COMP_ADDER;
				break;
			}
			
			case(OP_MULT):
			{
				hw_instr -> function = COMP_MULTIPLIER;
				break;
			}
		}
		
		HardwareInstrInfo * hw_instr_info = get_hw_instr_info(hw_instr -> function);
		assert(input_count == hw_instr_info -> input_count);
		assert(output_count == hw_instr_info -> output_count);
		
	}
	
	mem_push_array(arena , HardwareInstr , hw_instr_count);
	
	CircuitAddress * primary_inputs = mem_end(*arena , CircuitAddress);
	u32 input_capacity = mem_rem(*arena , CircuitAddress);
	u32 input_count = 0;
	
	for(u32 pair_index = 0; pair_index < data_path_size; pair_index++)
	{
		
		DataPathPair * pair = data_path + pair_index;
		assert(pair -> instrA || pair -> instrB);
		
		if(!pair -> instrA)
		{
			assert(input_count < input_capacity);
			primary_inputs[input_count++] = CircuitAddress{pair -> min_bit_addr , pair -> max_bit_addr};
		}
	}
	
	mem_push_array(arena , CircuitAddress , input_count);
	
	CircuitAddress * primary_outputs = mem_end(*arena , CircuitAddress);
	u32 output_capacity = mem_rem(*arena , CircuitAddress);
	u32 output_count = 0;
	
	for(u32 pair_index = 0; pair_index < data_path_size; pair_index++)
	{
		
		DataPathPair * pair = data_path + pair_index;
		assert(pair -> instrA || pair -> instrB);
		
		if(!pair -> instrB)
		{
			assert(output_count < output_capacity);
			primary_outputs[output_count++] = {pair -> min_bit_addr , pair -> max_bit_addr};
		}
	}
	
	mem_push_array(arena , CircuitAddress , output_count);
	
	*primary_inputs_ptr = primary_inputs;
	*primary_outputs_ptr = primary_outputs;
	*primary_input_count_ptr = input_count;
	*primary_output_count_ptr = output_count;
	*hw_instrs_ptr = hw_instrs;
	*hw_instr_count_ptr = hw_instr_count; 
	
	return(curr_addr);
}



ControlUnitSpec allocate_ctr_unit_spec(Mem * arena , u32 ctr_line_capacity)
{
	
	ControlUnitSpec spec = {};
	
	spec.ctr_lines = mem_push_array(arena , u32 , ctr_line_capacity);
	spec.ctr_line_capacity = ctr_line_capacity;
	spec.ctr_line_count = 0;
	
	return(spec);
}

void start_data_path_spec(MipsSpec * spec , DataPathSpec * path , Mem * arena , u32 line_capacity , u32 ctr_line_capacity)
{
	
	path -> instr_spec_start = mem_size(spec -> instr_set , InstrSpec);
	//path -> line_set.lines = mem_push_array(arean , LineSpec , line_capacity);
	//path -> line_set.capacity = line_capacity;
	//path -> line_set.count = 0;
	
	path -> ctr_unit = allocate_ctr_unit_spec(arena , ctr_line_capacity);
}

void end_data_path_spec(MipsSpec * spec , DataPathSpec * path)
{
	path -> instr_spec_end = mem_size(spec -> instr_set , InstrSpec);
}

void start_data_path(CircuitProgram * prog , DataPathSpec * path)
{
	path -> hw_instr_start = prog -> hw_instr_count;
	path -> sim_mem_start = prog -> curr_byte_addr;
}

void end_data_path(CircuitProgram * prog , DataPathSpec * path)
{
	path -> hw_instr_end = prog -> hw_instr_count;
	path -> sim_mem_end = prog -> curr_byte_addr;
}


void generate_MIPS_data_path_spec(MipsSpec * spec , DataPathSpec * path , Mem * arena)
{
	
	u32 reg_width = spec -> reg_width;
	u32 reg_addr_width = spec -> reg_addr_width;
	
	Mem * lines = &spec -> lines;
	Mem * instr_set = &spec -> instr_set;
	Mem * ctr_lines = &spec -> instr_spec_ctr_line_arena;
	
	start_data_path_spec(spec , path , arena , 64 , 64);
	
	path -> enable_line = MIPS_PATH_ENABLE_LINE;
	
	push_line_spec(lines , MIPS_REG_OUTPUT_ADDR_1_LINE , "reg-output-addr-1" , reg_addr_width);
	push_line_spec(lines , MIPS_REG_OUTPUT_DATA_1_LINE , "reg-output-data-1" , reg_width);
	
	push_line_spec(lines , MIPS_REG_INPUT_ADDR_1_LINE , "reg-input-addr-1" , reg_addr_width);
	push_line_spec(lines , MIPS_REG_INPUT_DATA_1_LINE , "reg-input-data-1" , reg_width);
	
	push_line_spec(lines , MIPS_REG_INPUT_ADDR_2_LINE , "reg-input-addr-2" , reg_addr_width);
	push_line_spec(lines , MIPS_REG_INPUT_DATA_2_LINE , "reg-input-data-2" , reg_width);
	
	push_line_spec(lines , MIPS_REG_WRITE_ENABLE_LINE , "mips-reg-write-enable" , 1);
	push_line_spec(lines , MIPS_REG_WRITE_ADDR_LINE , "mips-reg-write-addr-enable" , reg_width);
	push_line_spec(lines , MIPS_REG_WRITE_DATA_LINE , "mips-reg-write-data-enable" , reg_width);
	
	push_line_spec(lines , MIPS_PC_WRITE_ENABLE_LINE , "mips-pc-write-enable" , 1);
	push_line_spec(lines , MIPS_BRANCH_LINE , "mips-branch" , 1);
	push_line_spec(lines , MIPS_JUMP_LINE , "mips-jump" , reg_width);
	
	push_line_spec(lines , MIPS_ALU_OUT_LINE , "alu-out" , reg_width);
	push_line_spec(lines , MIPS_ADDER_OUT_LINE , "adder-out" , reg_width);
	push_line_spec(lines , MIPS_SHIFT_LEFT_OUT_LINE , "shift-left-out" , reg_width);
	push_line_spec(lines , MIPS_SHIFT_RIGHT_OUT_LINE , "shift_right-out" , reg_width);
	push_line_spec(lines , MIPS_MULT_OUT_LINE , "mult-out" , reg_width);
	
	push_line_spec(lines , MIPS_PATH_ENABLE_LINE , "MIPS-path-enable" , 1);
	push_line_spec(lines , MIPS_ALU_FUNC_LINE , "alu-func" , 2);
	push_line_spec(lines , MIPS_WB_SEL_LINE , "wb-sel" , 1);
	push_line_spec(lines , MIPS_DATA_MEM_READ_LINE , "data-mem-read" , 1);
	push_line_spec(lines , MIPS_DATA_MEM_WRITE_LINE , "data-mem-write" , 1);
	push_line_spec(lines , MIPS_IMM_SEL_LINE , "imm-sel" , 1);
	push_line_spec(lines , MIPS_REG_WRITE_SEL_LINE , "reg-write-sel" , 1);
	
	u32 ctr_line_capacity = 16;
	InstrSpec * instr = 0;
	
	assert(OP_HALT == 0);
	instr = push_instr_spec(path , instr_set , ctr_lines , ctr_line_capacity , "HALT" , OP_HALT , INSTR_R_TYPE);
	set_ctr_line(instr , MIPS_PATH_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_REG_WRITE_ENABLE_LINE , 0);
	set_ctr_line(instr , MIPS_PC_WRITE_ENABLE_LINE , 0);
	set_ctr_line(instr , MIPS_BRANCH_LINE , 0);
	set_ctr_line(instr , MIPS_ALU_FUNC_LINE , ALU_ADD);
	set_ctr_line(instr , MIPS_WB_SEL_LINE , WB_SEL_ALU);
	set_ctr_line(instr , MIPS_IMM_SEL_LINE , 0);
	set_ctr_line(instr , MIPS_REG_WRITE_SEL_LINE , 0);
	set_ctr_line(instr , MIPS_DATA_MEM_READ_LINE , 0);
	set_ctr_line(instr , MIPS_DATA_MEM_WRITE_LINE , 0);
	
	instr = push_instr_spec(path , instr_set , ctr_lines , ctr_line_capacity , "ADD" , OP_ADD , INSTR_R_TYPE);
	set_ctr_line(instr , MIPS_PATH_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_REG_WRITE_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_PC_WRITE_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_BRANCH_LINE , 0);
	set_ctr_line(instr , MIPS_ALU_FUNC_LINE , ALU_ADD);
	set_ctr_line(instr , MIPS_WB_SEL_LINE , WB_SEL_ALU);
	set_ctr_line(instr , MIPS_IMM_SEL_LINE , 0);
	set_ctr_line(instr , MIPS_REG_WRITE_SEL_LINE , 0);
	set_ctr_line(instr , MIPS_DATA_MEM_READ_LINE , 0);
	set_ctr_line(instr , MIPS_DATA_MEM_WRITE_LINE , 0);
	
	instr = push_instr_spec(path , instr_set , ctr_lines , ctr_line_capacity , "ADDI" , OP_ADD_I , INSTR_I_TYPE);
	set_ctr_line(instr , MIPS_PATH_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_REG_WRITE_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_PC_WRITE_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_BRANCH_LINE , 0);
	set_ctr_line(instr , MIPS_ALU_FUNC_LINE , ALU_ADD);
	set_ctr_line(instr , MIPS_WB_SEL_LINE , WB_SEL_ALU);
	set_ctr_line(instr , MIPS_IMM_SEL_LINE , 1);
	set_ctr_line(instr , MIPS_REG_WRITE_SEL_LINE , 1);
	set_ctr_line(instr , MIPS_DATA_MEM_READ_LINE , 0);
	set_ctr_line(instr , MIPS_DATA_MEM_WRITE_LINE , 0);
	
	instr = push_instr_spec(path , instr_set , ctr_lines , ctr_line_capacity , "MULT" , OP_MULT , INSTR_R_TYPE);
	set_ctr_line(instr , MIPS_PATH_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_REG_WRITE_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_PC_WRITE_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_BRANCH_LINE , 0);
	set_ctr_line(instr , MIPS_ALU_FUNC_LINE , ALU_MULT);
	set_ctr_line(instr , MIPS_WB_SEL_LINE , WB_SEL_ALU);
	set_ctr_line(instr , MIPS_IMM_SEL_LINE , 0);
	set_ctr_line(instr , MIPS_DATA_MEM_READ_LINE , 0);
	set_ctr_line(instr , MIPS_DATA_MEM_WRITE_LINE , 0);
	
	instr = push_instr_spec(path , instr_set , ctr_lines , ctr_line_capacity , "SLA" , OP_SHIFT_LEFT , INSTR_I_TYPE);
	set_ctr_line(instr , MIPS_PATH_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_REG_WRITE_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_PC_WRITE_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_BRANCH_LINE , 0);
	set_ctr_line(instr , MIPS_ALU_FUNC_LINE , ALU_SHIFT_LEFT);
	set_ctr_line(instr , MIPS_WB_SEL_LINE , WB_SEL_ALU);
	set_ctr_line(instr , MIPS_IMM_SEL_LINE , 1);
	set_ctr_line(instr , MIPS_REG_WRITE_SEL_LINE , 1);
	set_ctr_line(instr , MIPS_DATA_MEM_READ_LINE , 0);
	set_ctr_line(instr , MIPS_DATA_MEM_WRITE_LINE , 0);
	
	instr = push_instr_spec(path , instr_set , ctr_lines , ctr_line_capacity , "SRA" , OP_SHIFT_RIGHT , INSTR_I_TYPE);
	set_ctr_line(instr , MIPS_PATH_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_REG_WRITE_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_PC_WRITE_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_BRANCH_LINE , 0);
	set_ctr_line(instr , MIPS_ALU_FUNC_LINE , ALU_SHIFT_RIGHT);
	set_ctr_line(instr , MIPS_WB_SEL_LINE , WB_SEL_ALU);
	set_ctr_line(instr , MIPS_IMM_SEL_LINE , 1);
	set_ctr_line(instr , MIPS_REG_WRITE_SEL_LINE , 1);
	set_ctr_line(instr , MIPS_DATA_MEM_READ_LINE , 0);
	set_ctr_line(instr , MIPS_DATA_MEM_WRITE_LINE , 0);
	
	instr = push_instr_spec(path , instr_set , ctr_lines , ctr_line_capacity , "J" , OP_JUMP , INSTR_J_TYPE);
	set_ctr_line(instr , MIPS_PATH_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_REG_WRITE_ENABLE_LINE , 0);
	set_ctr_line(instr , MIPS_PC_WRITE_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_BRANCH_LINE , 1);
	set_ctr_line(instr , MIPS_ALU_FUNC_LINE , ALU_ADD);
	set_ctr_line(instr , MIPS_WB_SEL_LINE , WB_SEL_ALU);
	set_ctr_line(instr , MIPS_IMM_SEL_LINE , 0);
	set_ctr_line(instr , MIPS_REG_WRITE_SEL_LINE , 0);
	set_ctr_line(instr , MIPS_DATA_MEM_READ_LINE , 0);
	set_ctr_line(instr , MIPS_DATA_MEM_WRITE_LINE , 0);
	
	instr = push_instr_spec(path , instr_set , ctr_lines , ctr_line_capacity , "LW" , OP_LOAD , INSTR_I_TYPE);
	set_ctr_line(instr , MIPS_PATH_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_REG_WRITE_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_PC_WRITE_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_BRANCH_LINE , 0);
	set_ctr_line(instr , MIPS_ALU_FUNC_LINE , ALU_ADD);
	set_ctr_line(instr , MIPS_WB_SEL_LINE , WB_SEL_DATA_MEM);
	set_ctr_line(instr , MIPS_IMM_SEL_LINE , 1);
	set_ctr_line(instr , MIPS_REG_WRITE_SEL_LINE , 1);
	set_ctr_line(instr , MIPS_DATA_MEM_READ_LINE , 1);
	set_ctr_line(instr , MIPS_DATA_MEM_WRITE_LINE , 0);
	
	instr = push_instr_spec(path , instr_set , ctr_lines , ctr_line_capacity , "SW" , OP_STORE , INSTR_I_TYPE);
	set_ctr_line(instr , MIPS_PATH_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_REG_WRITE_ENABLE_LINE , 0);
	set_ctr_line(instr , MIPS_PC_WRITE_ENABLE_LINE , 1);
	set_ctr_line(instr , MIPS_BRANCH_LINE , 0);
	set_ctr_line(instr , MIPS_ALU_FUNC_LINE , ALU_ADD);
	set_ctr_line(instr , MIPS_WB_SEL_LINE , WB_SEL_ALU);
	set_ctr_line(instr , MIPS_IMM_SEL_LINE , 1);
	set_ctr_line(instr , MIPS_REG_WRITE_SEL_LINE , 0);
	set_ctr_line(instr , MIPS_DATA_MEM_READ_LINE , 0);
	set_ctr_line(instr , MIPS_DATA_MEM_WRITE_LINE , 1);
	
	end_data_path_spec(spec , path);
	do_spec_checks(spec);
}

void generate_MIPS_data_path(CircuitProgram * prog , MipsSpec * spec , DataPathSpec * path , Mem * arena)
{
	
	push_module(prog , "mips-path");
	
	start_data_path(prog , path);
	
	Mem line_arena = spec -> lines;
	Mem reg_arena = spec -> regs;
	Mem instr_spec_arena = spec -> instr_set;
	
	ControlUnitSpec ctr_unit_ = allocate_ctr_unit_spec(arena , 32);
	ControlUnitSpec * ctr_unit = &ctr_unit_;
	
	push_ctr_line(ctr_unit , MIPS_PATH_ENABLE_LINE);
	push_ctr_line(ctr_unit , MIPS_REG_WRITE_ENABLE_LINE);
	push_ctr_line(ctr_unit , MIPS_PC_WRITE_ENABLE_LINE);
	push_ctr_line(ctr_unit , MIPS_BRANCH_LINE);
	push_ctr_line(ctr_unit , MIPS_ALU_FUNC_LINE);
	push_ctr_line(ctr_unit , MIPS_WB_SEL_LINE);
	push_ctr_line(ctr_unit , MIPS_DATA_MEM_READ_LINE);
	push_ctr_line(ctr_unit , MIPS_DATA_MEM_WRITE_LINE);
	push_ctr_line(ctr_unit , MIPS_IMM_SEL_LINE);
	push_ctr_line(ctr_unit , MIPS_REG_WRITE_SEL_LINE);
	
	do_ctr_spec_checks(line_arena , ctr_unit);
	
	// NOTE: path inputs
	
	//RegisterSpec * regs = spec -> registers;
	//u32 reg_count;
	
	CircuitAddress clock = get_line_addr(line_arena , CLOCK_LINE);
	//CircuitAddress fetch_line = get_line_addr(line_arena , FETCH_STAGE_LINE);
	//CircuitAddress decode_line = get_line_addr(line_arena , DECODE_STAGE_LINE);
	CircuitAddress execute_line = get_line_addr(line_arena , EXECUTE_STAGE_LINE);
	
	CircuitAddress op_code_in = get_line_addr(line_arena , IR_OP_CODE_LINE);
	CircuitAddress rs_in = get_line_addr(line_arena , IR_RS_LINE);
	CircuitAddress rt_in = get_line_addr(line_arena , IR_RT_LINE);
	CircuitAddress rd_in = get_line_addr(line_arena , IR_RD_LINE);
	CircuitAddress imm_in = get_line_addr(line_arena , IR_IMM_LINE);
	CircuitAddress jump_in = get_line_addr(line_arena , IR_JUMP_LINE);
	
	// NOTE: path outputs
	
	CircuitAddress reg_write_enable_out = get_line_addr(line_arena , REG_WRITE_ENABLE_LINE);
	CircuitAddress reg_write_addr_out = get_line_addr(line_arena , REG_WRITE_ADDR_LINE);
	CircuitAddress reg_write_data_out = get_line_addr(line_arena , REG_WRITE_DATA_LINE);
	CircuitAddress pc_write_enable_out = get_line_addr(line_arena , PC_WRITE_ENABLE_LINE);
	CircuitAddress branch_out = get_line_addr(line_arena , BRANCH_LINE);
	CircuitAddress jump_out = get_line_addr(line_arena , JUMP_LINE);
	CircuitAddress path_enable = get_line_addr(line_arena , MIPS_PATH_ENABLE_LINE);
	
	// NOTE: control lines
	
	CircuitAddress reg_write_enable = get_line_addr(line_arena , MIPS_REG_WRITE_ENABLE_LINE);
	CircuitAddress pc_write_enable = get_line_addr(line_arena , MIPS_PC_WRITE_ENABLE_LINE);
	//CircuitAddress path_enable = get_line_addr(line_arena , MIPS_PATH_ENABLE_LINE);
	CircuitAddress branch = get_line_addr(line_arena , MIPS_BRANCH_LINE);
	CircuitAddress alu_func = get_line_addr(line_arena , MIPS_ALU_FUNC_LINE);
	CircuitAddress reg_write_sel = get_line_addr(line_arena , MIPS_REG_WRITE_SEL_LINE);
	CircuitAddress wb_sel = get_line_addr(line_arena , MIPS_WB_SEL_LINE);
	CircuitAddress imm_sel = get_line_addr(line_arena , MIPS_IMM_SEL_LINE);
	CircuitAddress data_mem_read = get_line_addr(line_arena , MIPS_DATA_MEM_READ_LINE);
	CircuitAddress data_mem_write = get_line_addr(line_arena , MIPS_DATA_MEM_WRITE_LINE);
	
	// NOTE: data lines
	
	CircuitAddress reg_output_addr_1 = get_line_addr(line_arena , MIPS_REG_OUTPUT_ADDR_1_LINE);
	CircuitAddress reg_output_data_1 = get_line_addr(line_arena , MIPS_REG_OUTPUT_DATA_1_LINE);
	
	CircuitAddress reg_input_addr_1 = get_line_addr(line_arena , MIPS_REG_INPUT_ADDR_1_LINE);
	CircuitAddress reg_input_data_1 = get_line_addr(line_arena , MIPS_REG_INPUT_DATA_1_LINE);
	
	CircuitAddress reg_input_addr_2 = get_line_addr(line_arena , MIPS_REG_INPUT_ADDR_2_LINE);
	CircuitAddress reg_input_data_2 = get_line_addr(line_arena , MIPS_REG_INPUT_DATA_2_LINE);
	
	CircuitAddress alu_out = get_line_addr(line_arena , MIPS_ALU_OUT_LINE);
	CircuitAddress mult_out = get_line_addr(line_arena , MIPS_MULT_OUT_LINE);
	CircuitAddress adder_out = get_line_addr(line_arena , MIPS_ADDER_OUT_LINE);
	CircuitAddress shift_left_out = get_line_addr(line_arena , MIPS_SHIFT_LEFT_OUT_LINE);
	CircuitAddress shift_right_out = get_line_addr(line_arena , MIPS_SHIFT_RIGHT_OUT_LINE);
	
	u32 reg_width = spec -> reg_width;
	
	CircuitAddress imm_mux_out = push_addr(prog , "imm_mux_out" , reg_width , 0);
	CircuitAddress data_mem_out = push_addr(prog , "data_mem_out" , reg_width , 0);
	
	connect(prog , rs_in , reg_input_addr_1);
	connect(prog , rt_in , reg_input_addr_2);
	push_mux2(prog , reg_output_addr_1 , rd_in , rt_in , reg_write_sel);
	
	push_transmission_gate(prog , reg_write_data_out , reg_output_data_1 , path_enable);
	push_transmission_gate(prog , reg_write_addr_out , reg_output_addr_1 , path_enable);
	push_transmission_gate(prog , reg_write_enable_out , reg_write_enable , path_enable);
	push_transmission_gate(prog , pc_write_enable_out , pc_write_enable , path_enable);
	push_transmission_gate(prog , branch_out , branch , path_enable);
	push_transmission_gate(prog , jump_out , jump_in , path_enable);
	
	push_module(prog , "input-decoders");
	
	generate_load_decoder(
						  prog , 
						  line_arena , reg_arena , 
						  reg_input_data_1 , reg_input_addr_1 , arena
						  );
	
	generate_load_decoder(
						  prog , 
						  line_arena , reg_arena , 
						  reg_input_data_2 , reg_input_addr_2 , arena
						  );
	
	pop_module(prog);
	
	push_module(prog , "ALU");
	MIPS_generate_ALU(
					  prog , reg_width , clock , 
					  alu_out , adder_out , mult_out , shift_left_out , shift_right_out , 
					  reg_input_data_1 , imm_mux_out , alu_func
					  );
	
	pop_module(prog);
	
	push_module(prog , "D-mem");
	MIPS_generate_data_mem(
						   prog , prog -> ports , prog -> port_count , 
						   clock , execute_line , 
						   data_mem_out , alu_out , reg_input_data_2 , 
						   data_mem_read , data_mem_write
						   );
	pop_module(prog);
	
	push_module(prog , "imm mux");
	push_mux2(prog , imm_mux_out , reg_input_data_2 , imm_in , imm_sel);
	pop_module(prog);
	
	push_module(prog , "writeback mux");
	push_mux2(prog , reg_output_data_1 , alu_out , data_mem_out , wb_sel);
	pop_module(prog);
	
	push_module(prog , "control-unit");
	MIPS_generate_control_unit(
							   prog , arena , 
							   line_arena , instr_spec_arena , path -> instr_spec_start , path -> instr_spec_end , 
							   ctr_unit , 
							   clock , op_code_in
							   );
	pop_module(prog);
	
	pop_module(prog);
	end_data_path(prog , path);
}

void generate_DOT4_data_path_spec(MipsSpec * spec , DataPathSpec * path , Mem * arena)
{
	
	//u32 reg_width = spec -> reg_width;
	
	Mem * lines = &spec -> lines;
	Mem * instr_set = &spec -> instr_set;
	Mem * ctr_lines = &spec -> instr_spec_ctr_line_arena;
	
	start_data_path_spec(spec , path , arena , 64 , 64);
	
	path -> enable_line = DOT4_PATH_ENABLE_LINE;
	push_line_spec(lines , DOT4_PATH_ENABLE_LINE , "dot4-path-enable" , 1);	
	push_line_spec(lines , DOT4_REG_WRITE_ENABLE_LINE , "dot4-reg-write-enable" , 1);	
	
	u32 ctr_line_capacity = 16;
	InstrSpec * instr = 0;
	
	instr = push_instr_spec(path , instr_set , ctr_lines , ctr_line_capacity , "DOT4" , OP_DOT4 , INSTR_R_TYPE);
	set_ctr_line(instr , DOT4_PATH_ENABLE_LINE , 1);
	set_ctr_line(instr , DOT4_REG_WRITE_ENABLE_LINE , 1);
}

void generate_DOT4_data_path(CircuitProgram * prog , MipsSpec * spec , DataPathSpec * path , Mem * arena)
{
	
	push_module(prog , "dot4-path");
	start_data_path(prog , path);
	
	u32 reg_width = spec -> reg_width;
	
	Mem line_arena = spec -> lines;
	Mem reg_arena = spec -> regs;
	Mem instr_spec_arena = spec -> instr_set;
	
	ControlUnitSpec ctr_unit_ = allocate_ctr_unit_spec(arena , 32);
	ControlUnitSpec * ctr_unit = &ctr_unit_;
	
	push_ctr_line(ctr_unit , DOT4_PATH_ENABLE_LINE);
	push_ctr_line(ctr_unit , DOT4_REG_WRITE_ENABLE_LINE);
	
	do_ctr_spec_checks(line_arena , ctr_unit);
	
	CircuitAddress clock = get_line_addr(line_arena , CLOCK_LINE);
	CircuitAddress op_code_in = get_line_addr(line_arena , IR_OP_CODE_LINE);
	CircuitAddress rd_in = get_line_addr(line_arena , IR_RD_LINE);
	CircuitAddress reg_write_data_out = get_line_addr(line_arena , REG_WRITE_DATA_LINE);
	CircuitAddress reg_write_addr_out = get_line_addr(line_arena , REG_WRITE_ADDR_LINE);
	CircuitAddress path_enable = get_line_addr(line_arena , DOT4_PATH_ENABLE_LINE);
	
	CircuitAddress A1 = get_reg_addr(reg_arena , line_arena , "r1");
	CircuitAddress A2 = get_reg_addr(reg_arena , line_arena , "r2");
	CircuitAddress A3 = get_reg_addr(reg_arena , line_arena , "r3");
	CircuitAddress A4 = get_reg_addr(reg_arena , line_arena , "r4");
	
	CircuitAddress B1 = get_reg_addr(reg_arena , line_arena , "r5");
	CircuitAddress B2 = get_reg_addr(reg_arena , line_arena , "r6");
	CircuitAddress B3 = get_reg_addr(reg_arena , line_arena , "r7");
	CircuitAddress B4 = get_reg_addr(reg_arena , line_arena , "r8");
	
	CircuitAddress AB1 = push_addr(prog , reg_width);
	CircuitAddress AB2 = push_addr(prog , reg_width);
	CircuitAddress AB3 = push_addr(prog , reg_width);
	CircuitAddress AB4 = push_addr(prog , reg_width);
	
	CircuitAddress AB12 = push_addr(prog , reg_width);
	CircuitAddress AB34 = push_addr(prog , reg_width);
	CircuitAddress AB1234 = push_addr(prog , reg_width);
	
	push_module(prog , "dot-unit");
	push_multiplier(prog , AB1 , A1 , B1);
	push_multiplier(prog , AB2 , A2 , B2);
	push_multiplier(prog , AB3 , A3 , B3);
	push_multiplier(prog , AB4 , A4 , B4);
	
	push_adder(prog , AB12 , AB1 , AB2);
	push_adder(prog , AB34 , AB3 , AB4);
	push_adder(prog , AB1234 , AB12 , AB34);
	
	push_transmission_gate(prog , reg_write_data_out , AB1234 , path_enable);
	push_transmission_gate(prog , reg_write_addr_out ,  rd_in , path_enable);
	pop_module(prog);
	
	push_module(prog , "control-unit");
	MIPS_generate_control_unit(
							   prog , arena , 
							   line_arena , instr_spec_arena , path -> instr_spec_start , path -> instr_spec_end , 
							   ctr_unit , 
							   clock , op_code_in
							   );
	pop_module(prog);
	
	pop_module(prog);
	
	end_data_path(prog , path);
}

void define_mips_spec(MipsSpec * spec , Mem * arena)
{
	
	Mem * lines = &spec -> lines;
	Mem * fields = &spec -> fields;
	Mem * stage_lines = &spec -> stage_lines;
	Mem * regs = &spec -> regs;
	Mem * ports = &spec -> ports;
	
	u32 instr_width = 32;
	u32 reg_width = 32;
	u32 reg_addr_width = 5;
	
	spec -> instr_width = instr_width;
	spec -> reg_width = reg_width;
	spec -> reg_addr_width = reg_addr_width;
	spec -> max_imm_value = (1 << 16) - 1;
	spec -> min_jump_value = - ((1 << 25) - 1);
	spec -> max_jump_value =    (1 << 25) - 1;
	
	push_line_spec(lines , CLOCK_LINE , "clock" , 1);
	push_line_spec(lines , PC_LINE , "PC" , 32);
	push_line_spec(lines , IR_LINE , "IR" , instr_width);
	push_line_spec(lines , REG_WRITE_ENABLE_LINE , "reg-write-enable" , 1);
	push_line_spec(lines , REG_WRITE_ADDR_LINE , "reg-write-addr" , reg_addr_width);
	push_line_spec(lines , REG_WRITE_DATA_LINE , "reg-write-data" , reg_width);
	push_line_spec(lines , PC_WRITE_ENABLE_LINE , "pc-write-enable" , 1);
	push_line_spec(lines , BRANCH_LINE , "branch" , 1);
	push_line_spec(lines , JUMP_LINE , "jump" , 32);
	
	push_line_spec(lines , IR_IMM_LINE , "IR-imm" , reg_width);
	push_line_spec(lines , IR_JUMP_LINE , "IR-jump" , reg_width);
	
	push_instr_field_spec(fields , lines , FIELD_CONSTANT , IR_OP_CODE_LINE , "IR-op-code" , 26 , 32 , 0);
	push_instr_field_spec(fields , lines , FIELD_REGISTER , IR_RS_LINE , "IR-rs" , 21 , 26, 0);
	push_instr_field_spec(fields , lines , FIELD_REGISTER , IR_RT_LINE ,"IR-rt" , 16 , 21 , 0);
	push_instr_field_spec(fields , lines , FIELD_REGISTER , IR_RD_LINE , "IR-rd" , 11 , 16 , 0);
	push_instr_field_spec(fields , lines , FIELD_CONSTANT , IR_IMM_LINE_DIRECT , "IR-imm-direct" , 0 , 16 , 1);
	push_instr_field_spec(fields , lines , FIELD_CONSTANT , IR_JUMP_LINE_DIRECT , "IR-jump-direct" , 0 , 26 , 1);
	
	push_stage_line_spec(stage_lines , lines , FETCH_STAGE_LINE , INSTR_STAGE_FETCH , "stage-fetch" , 1);
	push_stage_line_spec(stage_lines , lines , DECODE_STAGE_LINE , INSTR_STAGE_DECODE , "stage-decode" , 1);
	push_stage_line_spec(stage_lines , lines , EXECUTE_STAGE_LINE , INSTR_STAGE_EXECUTE , "stage-execute" , 1);
	
	u32 reg_line_id = MAX_LINE_ENUM+1;
	u32 reg_select_addr = 0;
	
	push_register_spec(regs , lines , ZR_REGISTER , reg_line_id++ , "r0" , reg_width , reg_select_addr++ , 0);
	push_register_spec(regs , lines , GP_REGISTER , reg_line_id++ , "r1" , reg_width , reg_select_addr++ , 0);
	push_register_spec(regs , lines , GP_REGISTER , reg_line_id++ , "r2" , reg_width , reg_select_addr++ , 0);
	push_register_spec(regs , lines , GP_REGISTER , reg_line_id++ , "r3" , reg_width , reg_select_addr++ , 0);
	push_register_spec(regs , lines , GP_REGISTER , reg_line_id++ , "r4" , reg_width , reg_select_addr++ , 0);
	push_register_spec(regs , lines , GP_REGISTER , reg_line_id++ , "r5" , reg_width , reg_select_addr++ , 0);
	push_register_spec(regs , lines , GP_REGISTER , reg_line_id++ , "r6" , reg_width , reg_select_addr++ , 0);
	push_register_spec(regs , lines , GP_REGISTER , reg_line_id++ , "r7" , reg_width , reg_select_addr++ , 0);
	push_register_spec(regs , lines , GP_REGISTER , reg_line_id++ , "r8" , reg_width , reg_select_addr++ , 0);
	push_register_spec(regs , lines , GP_REGISTER , reg_line_id++ , "r9" , reg_width , reg_select_addr++ , 0);
	push_register_spec(regs , lines , GP_REGISTER , reg_line_id++ , "r10" , reg_width , reg_select_addr++ , 0);
	
	// TODO: 
	u32 port_min = (u32)MIN_SIGNED_VALUE(16);
	u32 port_max = (u32)MAX_SIGNED_VALUE(16);
	
	push_port_spec(ports , lines , PORT0_LINE , "port0" , "port0" , 0 , reg_width , 1 , port_min , port_max);
	push_port_spec(ports , lines , PORT1_LINE , "port1" , "port1" , 1 , reg_width , 1 , port_min , port_max);
	
	DataPathSpec * mips_path = push_data_path_spec(&spec -> paths , MIPS_PATH , 16 , arena);
	DataPathSpec * dot4_path = push_data_path_spec(&spec -> paths , DOT4_PATH , 16 , arena);
	
	generate_MIPS_data_path_spec(spec , mips_path , arena);
	generate_DOT4_data_path_spec(spec , dot4_path , arena);
	
	do_spec_checks(spec);
	
}

void allocate_lines(CircuitProgram * prog , Mem line_arena)
{
	
	LineSpec * lines = mem_start(line_arena , LineSpec);
	u32 line_count = mem_size(line_arena , LineSpec);
	
	for(u32 line_index = 0; line_index < line_count; line_index++)
	{
		LineSpec * curr_line = lines + line_index;
		curr_line -> addr = push_addr(prog , curr_line -> name , curr_line -> width , 0);
	}
	
	
}


void MIPS_generate(
				   CircuitProgram * prog , MipsSpec * spec , Instr * sw_instr , u32 sw_instr_count , Mem * arena)
{
	
	
	CircuitAddress gnd = push_addr(prog , "GND" , width_of(GND) , 0);
	CircuitAddress vdd = push_addr(prog , "VDD" , width_of(VDD) , 0);
	
	push_hw_instruction(prog , UPDATE_IO , 0 , 0);
	push_input_vector(prog , GND , CircuitValue{0x00000000 , width_of(GND)});
	push_input_vector(prog , VDD , CircuitValue{0xFFFFFFFF , width_of(VDD)});
	
	assert(gnd == GND);
	assert(vdd == VDD);
	
	Mem lines = spec -> lines;
	Mem stage_arena = spec -> stage_lines;
	Mem register_arena = spec -> regs;
	Mem field_arena = spec -> fields;
	Mem port_arena = spec -> ports;
	Mem path_arena = spec -> ports;
	u32 reg_width = spec -> reg_width;
	
	//MipsSpec * dot_spec = define_dot_spec();
	
	CircuitAddress core_addr_space = {};
	core_addr_space.min = prog -> curr_byte_addr*8;	
	
	allocate_lines(prog , lines);
	MIPS_generate_ports(prog , lines , port_arena);
	
	CircuitAddress clock = get_line_addr(lines , CLOCK_LINE);
	CircuitAddress fetch_line = get_line_addr(lines , FETCH_STAGE_LINE);
	//CircuitAddress decode_line = get_line_addr(lines , DECODE_STAGE_LINE);
	CircuitAddress execute_line = get_line_addr(lines , EXECUTE_STAGE_LINE);
	CircuitAddress IR = get_line_addr(lines , IR_LINE);
	CircuitAddress PC = get_line_addr(lines , PC_LINE);
	
	CircuitAddress reg_write_enable = get_line_addr(lines , REG_WRITE_ENABLE_LINE);
	CircuitAddress reg_write_addr = get_line_addr(lines , REG_WRITE_ADDR_LINE);
	CircuitAddress reg_write_data = get_line_addr(lines , REG_WRITE_DATA_LINE);
	
	CircuitAddress pc_write_enable = get_line_addr(lines , PC_WRITE_ENABLE_LINE);
	CircuitAddress branch = get_line_addr(lines , BRANCH_LINE);
	CircuitAddress jump = get_line_addr(lines , IR_JUMP_LINE);
	
	CircuitAddress IR_imm_direct = get_line_addr(lines , IR_IMM_LINE_DIRECT);
	CircuitAddress IR_jump_direct = get_line_addr(lines , IR_JUMP_LINE_DIRECT);
	CircuitAddress IR_imm = get_line_addr(lines , IR_IMM_LINE);
	CircuitAddress IR_jump = get_line_addr(lines , IR_JUMP_LINE);
	CircuitAddress * regs = {};
	//u32 reg_count = mem_size(register_arena , RegisterSpec);
	
	//CircuitAddress * stage_lines = {};
	//u32 stage_line_count = mem_size(stage_arena , StageLineSpec);
	
	push_sign_ext(prog , spec , IR_imm , IR_imm_direct , 1);
	push_sign_ext(prog , spec , IR_jump , IR_jump_direct , 1);
	
	
	MIPS_make_IR_field_connections(prog , field_arena , lines , IR);
	
	// NOTE: output busses
	push_module(prog , "stage_counter");
	MIPS_generate_stage_counter(
								prog , stage_arena , lines,  clock , arena
								);
	pop_module(prog);
	
	push_module(prog , "register file");
	MIPS_generate_register_file(
								prog , arena , 
								lines , register_arena , port_arena , 
								reg_width , regs , 
								clock , execute_line , reg_write_enable , 
								reg_write_addr , reg_write_data
								);
	pop_module(prog);
	
	
	push_module(prog , "I-mem");
	MIPS_generate_instruction_memory(
									 prog , sw_instr , sw_instr_count , 
									 clock , fetch_line , IR , PC
									 );
	pop_module(prog);
	
	push_module(prog , "PC-logic");
	MIPS_generate_PC_logic(
						   prog , clock , execute_line , pc_write_enable , 
						   PC , jump , branch
						   );
	pop_module(prog);
	
	DataPathSpec * data_paths = mem_start(path_arena , DataPathSpec);
	generate_MIPS_data_path(prog , spec , data_paths + 0 , arena);
	generate_DOT4_data_path(prog , spec , data_paths + 1 , arena);
	
	core_addr_space.max = prog -> curr_byte_addr*8;	
	MIPS_generate_clock_logic(prog , clock , core_addr_space);
	
	check_integrity(prog , spec);
	
	
}



inline
u32 get_static_energy()
{
	
}

inline
u32 get_norm_switch_energy(CircuitAddressPower power)
{
	return(power.fanout * power.switches);
}

CircuitAddressPower * get_power(
								Mem addr_power_list_arena , 
								CircuitAddress target_addr , i32 time)
{
	
	assert_zero;
	CircuitAddressPower * result = 0;
	CircuitAddressPower * power_list = mem_start(addr_power_list_arena , CircuitAddressPower);
	u32 power_count = mem_capacity(addr_power_list_arena , CircuitAddressPower);
	
	u32 index = ((target_addr.min+1)*(target_addr.max+7)*(time+11)) % power_count;
	CircuitAddressPower * power = power_list + index;
	
	if(power -> addr == target_addr && power -> time == time)
	{
		result = power;
	}
	
	return(result);
}




void get_power_list_for_frame(CircuitEventTimeMark * time_mark , EventBuffer * event_buffer , Mem * addr_power_list_arena)
{
	
	CircuitPowerEvent * power_events = event_buffer -> power_events;
	CircuitFanoutEvent * fanout_events = event_buffer -> fanout_events;
	
	u32 fanout_capacity = event_buffer -> fanout_capacity;
	u32 power_capacity = event_buffer -> power_capacity;
	
	i32 time = time_mark -> time;
	
	u32 power_start_index = time_mark -> power_start_index;
	u32 power_end_index = time_mark -> power_end_index;
	u32 power_index = power_start_index;
	
	u32 fanout_start_index = time_mark -> fanout_start_index;
	u32 fanout_end_index = time_mark -> fanout_end_index;
	u32 fanout_index = fanout_start_index;
	
	//output_to_console("fanout % % %\n" , 0 , fanout_start_index , fanout_end_index , fanout_end_index-fanout_start_index);
	
	u32 total_f = 0;
	
	while(power_index != power_end_index)
	{
		
		CircuitAddressPower * power = mem_push_struct(addr_power_list_arena , CircuitAddressPower);
		CircuitPowerEvent * power_event = power_events + power_index;
		CircuitAddress out_addr = power_event -> addr;
		
		*power = {};
		
		u32 width = width_of(out_addr);
		u32 before = power_event -> before_value.v;
		u32 after = power_event -> after_value.v;
		u32 switch_bits = before ^ after;
		
		u32 fanout = 0;
		u32 switch_count = 0;
		
		fanout_start_index = time_mark -> fanout_start_index;
		fanout_end_index = time_mark -> fanout_end_index;
		fanout_index = fanout_start_index;
		
		while(fanout_index != fanout_end_index)
		{
			if(out_addr == fanout_events[fanout_index].addr)
			{
				fanout++;
				total_f++;
			}
			
			if(++fanout_index == fanout_capacity) fanout_index = 0;
		}
		
		for(u32 i = 0; i < width; i++)
		{
			switch_count += switch_bits & 0x1;
			switch_bits >>= 1;
		}
		
		power -> addr = out_addr;
		power -> time = time;
		power -> switches = switch_count;
		power -> fanout = fanout;
		
		if(++power_index == power_capacity) power_index = 0;
	}
	
	if(time == 17 || time == 18)
	{
		int x = 0; x++;
	}
	
	
}


void calculate_module_power(
							CircuitModule * module_list , u32 module_count , i32 time , 
							Mem addr_power_list_arena , Mem * module_power_list_arena)
{
	
	for(u32 module_index = 0; module_index < module_count; module_index++)
	{
		
		CircuitModule * module = module_list + module_index;
		CircuitModulePower * module_power = mem_push_struct(module_power_list_arena , CircuitModulePower);
		
		CircuitAddress * addr_list = module -> addr_list;
		u32 addr_count = module -> addr_count;
		
		module_power -> module_id = module -> id;
		module_power -> time = time;
		
		for(u32 addr_index = 0; addr_index < addr_count; addr_index++)
		{
			
			CircuitAddress module_addr = addr_list[addr_index];
			CircuitAddressPower * power = get_power(addr_power_list_arena , module_addr , time);
			
			if(power)
			{
				module_power -> norm_switch_energy += get_norm_switch_energy(*power);
				module_power -> switches += power -> switches;
				module_power -> fanout += power -> fanout;
			}
		}
	}
}

void calculate_frame_power(StatFrame * frame , i32 time , Mem addr_power_list_arena)
{
	
	CircuitAddressPower * power_list = mem_start(addr_power_list_arena , CircuitAddressPower);
	u32 power_count = mem_size(addr_power_list_arena , CircuitAddressPower);
	
	u32 total_norm_static_energy = 0;
	u32 total_norm_switch_energy = 0;
	u32 total_switches = 0;
	u32 total_fanout = 0;
	
	for(u32 power_index = 0; power_index < power_count; power_index++)
	{
		
		CircuitAddressPower * power = power_list + power_index;
		u32 switch_count = power -> switches;
		u32 fanout = power -> fanout;
		
		if(power -> time == time)
		{
			total_norm_switch_energy += get_norm_switch_energy(*power);
			total_switches += switch_count;
			total_fanout += fanout;
		}
	}
	
	if(time == 17 || time == 18)
	{
		int x = 0; x++;
	}
	
	
	
	frame -> time = time;
	frame -> norm_static_energy = total_norm_static_energy;
	frame -> norm_switch_energy = total_norm_switch_energy;
	frame -> switches = total_switches;
	frame -> fanout = total_fanout;
}

void perform_power_analysis(
							CircuitEventTimeMark * time_mark , StatTracker * stats , EventBuffer * event_buffer , 
							CircuitModule * module_list , u32 module_count)
{
	
	i32 time = time_mark -> time;
	i32 last_time = stats -> last_frame_time;
	
	Mem * addr_power_list_arena = &stats -> addr_power_arena;
	//Mem * module_power_list_arena = &stats -> module_power_arena; 
	
	StatFrame * frames = stats -> frames;
	u32 frame_count = stats -> frame_count;
	u32 frame_capacity = stats -> frame_capacity;
	
	assert(frame_count < frame_capacity);
	StatFrame * frame = frames + frame_count++;
	
	get_power_list_for_frame(
							 time_mark , event_buffer , 
							 addr_power_list_arena 
							 );
	
#if 0
	calculate_module_power(
						   module_list , module_count , time , 
						   *addr_power_list_arena , module_power_list_arena
						   );
#endif
	
	calculate_frame_power(
						  frame , time , *addr_power_list_arena
						  );
	
	u32 comb_energy = frame -> norm_static_energy + frame -> norm_switch_energy + frame -> norm_short_circuit_energy;
	u32 total_switches = frame -> switches;
	
	stats -> total_energy += comb_energy;
	stats -> total_switches += total_switches;
	stats -> max_energy = comb_energy > stats -> max_energy ? comb_energy : stats -> max_energy;
	stats -> max_switches = total_switches > stats -> max_switches ? total_switches : stats -> max_switches;
	
	stats -> frame_count = frame_count;
	stats -> frames_recorded += (time != last_time);
	
}

void perform_power_analysis(
							StatTracker * stats , EventBuffer * event_buffer , 
							CircuitModule * module_list , u32 module_count)
{
	
	CircuitEventTimeMark * marks = event_buffer -> time_marks;
	u32 mark_capacity = event_buffer -> time_mark_capacity;
	u32 mark_end_index = event_buffer -> time_mark_end_index;
	u32 mark_index = event_buffer -> time_mark_start_index;
	
	i32 last_frame_time = stats -> last_frame_time;
	i32 max_frame_time = last_frame_time;
	
	mem_pop(&stats -> addr_power_arena , stats -> addr_power_arena.base);
	
	while(mark_index != mark_end_index)
	{
		
		CircuitEventTimeMark * time_mark = marks + mark_index;
		if(time_mark -> time > last_frame_time)
		{
			perform_power_analysis(
								   time_mark , stats , event_buffer , 
								   module_list , module_count
								   );
		}
		
		if(time_mark -> time > max_frame_time)
		{
			max_frame_time = time_mark -> time;
		}
		
		if(++mark_index == mark_capacity) mark_index = 0;
	}
	
	stats -> last_frame_time = max_frame_time;
}


void perform_stage_analysis(StatTracker * stats , InstrTracker * instrs)
{
	
	
	InstrStage * stage_list = instrs -> stage_list;
	u32 stage_count = instrs -> stage_count;
	
	CompletedInstr * completed_instrs = instrs -> completed_instrs;
	u32 completed_instr_count = instrs -> completed_instr_count;
	
	u32 stages_recorded = stats -> stages_recorded;
	i32 last_stage_time = stats -> last_stage_time;
	i32 next_stage_time = stats -> last_stage_time;
	
	u32 instrs_recorded = stats -> instrs_recorded;
	i32 last_instr_time = stats -> last_stage_time;
	i32 next_instr_time = stats -> last_stage_time;
	
	u32 total_period = stats -> total_stage_period;
	u32 max_period = stats -> max_stage_period;
	
	for(u32 stage_index = 0; stage_index < stage_count; stage_index++)
	{
		
		InstrStage * stage = stage_list + stage_index;
		i32 time = stage -> time;
		i32 period = stage -> period;
		
		if(last_stage_time < time && stage -> state == 2)
		{
			total_period += period;
			max_period = max_period < period ? period : max_period;
			next_stage_time = next_stage_time < time ? time : next_stage_time;
			stages_recorded++;
		}
	}
	
	for(u32 instr_index = 0; instr_index < completed_instr_count; instr_index++)
	{
		CompletedInstr * instr = completed_instrs + instr_index;
		i32 time = instr -> time;
		
		if(last_instr_time < time)
		{
			next_instr_time = next_instr_time < time ? time : next_instr_time;
			instrs_recorded++;
		}
	}
	
	stats -> stages_recorded = stages_recorded;
	stats -> instrs_recorded = instrs_recorded;
	stats -> total_stage_period = total_period;
	stats -> max_stage_period = max_period;
	stats -> last_stage_time = next_stage_time;
	stats -> last_instr_time = next_instr_time;
}


u32 write_register_name(char * buffer , u32 addr , Mem reg_spec_arena)
{
	
	char * curr_char = buffer;
	RegisterSpec * info = get_register_spec(reg_spec_arena , addr);
	
	if(info)
	{
		curr_char += str_format(curr_char , "\\$%" , info -> name);
	}
	
	return(curr_char-buffer);
}


u32 write_instruction(
					  char * buffer , Instr * instr , 
					  Mem lines , Mem instr_spec_arena , 
					  Mem instr_field_arena , Mem reg_spec_arena)
{
	
	char * curr_char = buffer;
	InstrSpec * info = get_instr_spec(instr_spec_arena , instr -> op_code);
	InstrFieldSpec fields[16] = {};
	u32 field_count = 0;
	
	if(!info) curr_char += str_format(curr_char , "??? %" , instr -> op_code);
	
	if(info)
	{
		curr_char = buffer;
		curr_char += str_format(curr_char , "% ${tabulate:10}" , info -> mnemonic);
		
		field_count = get_instr_fields(instr_field_arena , info -> type , fields);
		
		for(u32 field_index = 0; field_index < field_count; field_index++)
		{
			
			InstrFieldSpec field = fields[field_index];
			u32 type = field.type;
			u32 sign = field.sign;
			u32 value = get_field_value(*instr , field);
			
			switch(type)
			{
				case(FIELD_REGISTER):
				{
					curr_char += write_register_name(curr_char , instr -> rd , reg_spec_arena);
					break;
				}
				
				case(FIELD_CONSTANT):
				{
					curr_char += str_format(curr_char , "%" , sign ? (i32)value : (u32)value);
					break;
				}
				
				default:
				{
					curr_char += str_format(curr_char , "???");
				}
			}
			
			curr_char += str_copy(curr_char , " ");
		}
	}
	
	return(curr_char-buffer);
}

void print_instructions(
						Instr * instrs , u32 count , 
						Mem lines , Mem instr_spec_arena , 
						Mem instr_field_arena,  Mem reg_spec_arena)
{
	
	char buffer[128] = {};
	for(u32 i = 0; i < count; i++)
	{
		Instr * curr_instr = instrs + i;
		write_instruction(buffer , curr_instr , lines , instr_spec_arena , instr_field_arena , reg_spec_arena);
		printf("%s\n" , buffer);
	}
	
}



// TODO: put path inputs and output into path struct, to automate the transmission gates. 
// TODO: use the data path model to create the dot product and make a path with it.
// TODO: procedurally generated path tests
// TODO: ensure the all data path outputs are connected to something.
// TODO: contension checks.
// TODO: integrate ports
// TODO: power modeling.
// TODO: multi-port writes?

/*

  0:16  - imm
  0:26  - jump
  6:11  - shift
  11:16 - rd
  16:21 - rt
  21:26 - rs
  26:32 - op-code


*/

