/*
Author: Nathaniel Huesler
Year: 2025
Copywrite Notice: LICENSE.txt 
*/

/*

This file generates the user-interface. Buttons 1 to 4 select between tabs.
The right arrow in combination with control and shift keys execute the simulator by a 
set amount of instructions. 

<left>: execute the next hardware instruction
<ctr> <left>: execute a sim-pass. Execute until returning back the first instruction.
<ctr> <shift> <Left> : Executes the next assembly instruction. This will take many passes of the simulator.  
<space> Run the simulator indefintely. 
*/


#define TEXT_HEIGHT 14.0f
#define TEXT_COLOUR Vec4{0,0,0,1}

#define TABLE_LINE_COLOUR Vec4{0.6f , 0.6f , 0.6f , 1}
#define SELECTED_ROW_COLOUR Vec4{24.0f , 89.0f , 169.0f , 255.0f}/255.0f
#define INPUT_ADDR_ROW_COLOUR Vec4{40.0f , 230.0f , 40.0f , 255.0f}/255.0f
#define OUTPUT_ADDR_ROW_COLOUR Vec4{230.0f , 40.0f , 40.0f , 255.0f}/255.0f

#define BUTTON_UNPRESSED_COLOUR Vec4{124.0f , 142.0f , 188.0f , 255.0f}/255.0f
#define BUTTON_PRESSED_COLOUR Vec4{71.0f , 90.0f , 141.0f , 255.0f}/255.0f

#define TABLE_COLOUR_LAYER translation_mat4(Vec3{0,0,-1})
#define TABLE_TEXT_LAYER translation_mat4(Vec3{0,0,-2})
#define TABLE_OUTLINE_LAYER translation_mat4(Vec3{0,0,-3})

void debug_print_ey_state(KeyState key , char * name)
{
	char buffer[128] = {};
	u32 is_pressed = key_pressed(key);
	u32 is_released = key_released(key);
	f32 is_held = key_held(key , 0);
	
	str_format(
			   buffer , "key % ${tabulate:20} p% r% h%" , 
			   name , is_pressed , is_released , is_held
			   );
	output_string(buffer);
}

f32 scroll_in_region(f32 scroll , Rect region , InputState * input_state)
{
	
	MouseInput * mouse = &input_state -> mouse;
	Vec2 mouse_pos = {mouse -> pos_x , mouse -> pos_y};
	f32 scroll_wheel = mouse -> scroll_wheel;
	
	if(in_rect(mouse_pos , region))
	{
		scroll -= scroll_wheel;
	}
	
	return(scroll);
	
}

void do_controlls(
				  u32 * target_step_index_ptr , u32 * target_sw_instr_index_ptr , u32 * playing_ptr , 
				  u32 step_index , u32 sw_instr_index , u32 instr_count , 
				  u32 step_rate , u32 sw_instr_rate , 
				  InputState * input_state , f32 delta_time)
{
	
	u32 target_step_index = step_index;
	u32 target_sw_instr_index = sw_instr_index;
	u32 playing = *playing_ptr;
	
	KeyState left_key = input_state -> keys[KEY_LEFT];
	KeyState right_key = input_state -> keys[KEY_RIGHT];
	KeyState ctrl_key = input_state -> keys[KEY_CTRL];
	KeyState shift_key = input_state -> keys[KEY_SHIFT];
	KeyState space_key = input_state -> keys[KEY_SPACE];
	
	u32 ctrl_held = key_held(ctrl_key);
	u32 shift_held = key_held(shift_key);
	
	f32 hold_time = 0.3f;
	f32 hold_rate = 5.0f;
	f32 left_delta = typing_state(left_key , hold_time , hold_rate) > 0;
	f32 right_delta = typing_state(right_key , hold_time , hold_rate) > 0;
	i32 key_delta = (right_delta - left_delta);
	i32 delta = 0;
	
	playing = key_pressed(space_key) ? !playing : playing;
	
	if(playing)
	{
		target_step_index = step_index + step_rate;
		target_sw_instr_index = sw_instr_index + sw_instr_rate;
	}
	
	else
	{
		
		// TODO: only allow forward stepping until sim-reset is reintroduced.
		delta = max(delta , 0);
		target_step_index = step_index;
		target_sw_instr_index = sw_instr_index;
		
		if(ctrl_held && shift_held)
		{
			
			target_step_index += instr_count*999999;
			delta = key_delta;
			if(delta < 0 && -delta > target_sw_instr_index) 	target_sw_instr_index = 0;
			else 												target_sw_instr_index += delta;
		}
		
		else if(ctrl_held)
		{
			target_sw_instr_index += 999999;
			delta = key_delta*instr_count;
			if(delta < 0 && -delta > target_step_index)			target_step_index = 0;
			else												target_step_index += delta;
		}
		
		else
		{
			target_sw_instr_index += 999999;
			delta = key_delta;
			if(delta < 0 && -delta > target_step_index)			target_step_index = 0;
			else												target_step_index += delta;
		}
	}
	
	*target_step_index_ptr = target_step_index;
	*target_sw_instr_index_ptr = target_sw_instr_index;
	*playing_ptr = playing;
}


void do_info_ribbon(
					RenderBuffer * render_buffer , Font * font , InputState * input_state , Rect region , 
					u32 step_index , u32 next_step_index , u32 sw_instr_index , u32 next_sw_instr_index , 
					u32 instr_count , CircuitAddress selected_addr ,
					u32 * break_button_state , u32 * break_on_last_hw_instr_ptr , 
					Vec2 view_offset , f32 view_scale)
{
	
	
	u32 pass_index = step_index / instr_count;
	u32 next_pass_index = next_step_index / instr_count;
	
	u32 step_rate = next_step_index - step_index;
	u32 instr_rate = next_sw_instr_index - sw_instr_index;
	u32 pass_rate = next_pass_index - pass_index;
	
	push_clip(render_buffer , region);
	
	char buffer[256] = {};
	u32 break_on_last_hw_instr = 0;
	
	u32 buffer_size = str_format(
								 buffer , 
								 "step index: %{del:3} , step rate: %{del:3} / f , instr index: %{del:3} , instr rate: %{del:3} / f\n"
								 "pass index: %{del:3} , pass rate: %{del:3} / f , selected addr: [%:%]" , 
								 next_step_index , step_rate , sw_instr_index , instr_rate , 
								 pass_index , pass_rate , selected_addr.min , selected_addr.max
								 );
	
	f32 text_v_offset = (str_line_count(buffer , buffer_size)-1)*TEXT_HEIGHT;
	f32 text_width = get_text_width(buffer , TEXT_HEIGHT , font) + 10;
	Rect break_button_region = vertical_cut(
											horizontal_cut(region , text_width + 5.0f , 50.0f) , 
											0 , TEXT_HEIGHT + 5.0f
											);
	
	Vec2 text_pos = region.pos + Vec2{0 , text_v_offset};
	
	if(do_button_action(break_button_region , break_button_state , input_state))
	{
		break_on_last_hw_instr = 1;
	}
	
	push_text(
			  render_buffer , font , 1 , 
			  buffer , buffer_size , 
			  text_pos , text_pos.x , 
			  TEXT_HEIGHT , 0 , TEXT_COLOUR
			  );
	
	push_button(
				render_buffer , font , TEXT_HEIGHT , 
				break_button_region.pos , break_button_region.size , 
				"break" , *break_button_state , 
				BUTTON_UNPRESSED_COLOUR , BUTTON_PRESSED_COLOUR , TEXT_COLOUR
				);
	
	*break_on_last_hw_instr_ptr = break_on_last_hw_instr;
	
	pop_clip(render_buffer);
}

void get_relevant_instructions(
							   CircuitAddress selected_addr , 
							   HardwareInstr * instrs , u32 min_instr_index , u32 max_instr_index , 
							   u32 ** instr_indices_ptr , u32 * instr_index_count_ptr , u32 select_all , Mem * arena)
{
	
	u32 * instr_indices = mem_end(*arena , u32);
	u32 instr_index_capacity = mem_rem(*arena , u32);
	u32 instr_index_count = 0;
	
	if(!select_all)
	{
		for(u32 instr_index = min_instr_index; instr_index < max_instr_index; instr_index++)
		{
			HardwareInstr * instr = instrs + instr_index;
			HardwareInstrInfo * info = get_hw_instr_info(instr -> function);
			
			CircuitAddress * inputs = instr -> inputs;
			CircuitAddress * outputs = instr -> outputs;
			u32 input_count = info -> input_count;
			u32 output_count = info -> output_count;
			
			u32 valid = 1;
			
			if(selected_addr.min || selected_addr.max)
			{
				valid = 0;
				if(find_addr(selected_addr , inputs , input_count))
				{
					valid = 1;
				}
				
				else if (find_addr(selected_addr , outputs , output_count))
				{
					valid = 1;
				}
			}
			
			if(valid)
			{
				assert(instr_index_count < instr_index_capacity);
				instr_indices[instr_index_count++] = instr_index;
			}
		}
		
		mem_push_array(arena , u32 , instr_index_count);
		*instr_indices_ptr = instr_indices;
		*instr_index_count_ptr = instr_index_count;
	}
}

u32 address_is_viewable(
						CircuitAddress addr , CircuitAddress selected_addr , 
						HardwareInstr * instrs , u32 * relevant_instrs , u32 relevant_count , u32 select_all)
{
	
	u32 result = 1;
	
	if(!select_all)
	{
		
		result = 0;
		for( u32 instr_index = 0; 
			instr_index < relevant_count && !result;
			instr_index++)
		{
			
			u32 index = relevant_instrs[instr_index];
			HardwareInstr * instr = instrs + index;
			HardwareInstrInfo * info = get_hw_instr_info(instr -> function);
			
			CircuitAddress * inputs = instr -> inputs;
			CircuitAddress * outputs = instr -> outputs;
			
			u32 input_count = info -> input_count;
			u32 output_count = info -> output_count;
			
			CircuitAddress * input_addr = find_addr(addr , inputs , input_count);
			CircuitAddress * output_addr = find_addr(addr , outputs , output_count);
			
			if(input_addr || output_addr) result = 1;
		}
	}
	
	return(result);
}

void push_mem_row(
				  RenderBuffer * render_buffer , Font * font , u32 primary_key_pressed , Vec2 mouse_pos , Rect region , 
				  CircuitAddress selected_addr , CircuitAddress * next_selected_addr_ptr , 
				  Rect * row_sizer , f32 row_height , Rect addr_col , Rect prev_value_col , Rect curr_value_col , Rect next_value_col , 
				  CircuitState * prev_state , CircuitState * curr_state , CircuitState * next_state , 
				  CircuitAddress addr_of_interest , char * addr_buffer , 
				  CircuitAddress * curr_instr_inputs , u32 curr_instr_input_count ,
				  CircuitAddress * curr_instr_outputs , u32 curr_instr_output_count)
{
	
	
	f32 addr_to_pixels = row_height/8.0f;
	f32 margin_padding = 2.0f;
	
	char prev_value_buffer[128] = {};
	char curr_value_buffer[128] = {};
	char next_value_buffer[128] = {};
	
	CircuitAddress next_selected_addr = *next_selected_addr_ptr;
	//u32 byte_addr = addr_of_interest.min/8; 
	u32 base_bit_addr = (u32)(addr_of_interest.min/8)*8;
	u32 byte_width = get_byte_width(addr_of_interest);
	u32 is_selected = (addr_of_interest.min == selected_addr.min && addr_of_interest.max == selected_addr.max);
	
	Rect row = push_down(row_sizer , row_height*byte_width);
	Rect addr_cell = intersection(row , addr_col);
	CircuitAddress * input_addr_ptr = find_addr(
												addr_of_interest , curr_instr_inputs , curr_instr_input_count
												);
	
	CircuitAddress * output_addr_ptr = find_addr(
												 addr_of_interest , curr_instr_outputs , curr_instr_output_count
												 );
	
	if(primary_key_pressed && in_rect(mouse_pos , row))
	{
		
		if(is_selected)
		{
			next_selected_addr = {};
			is_selected = 0;
		}
		
		else
		{
			next_selected_addr = addr_of_interest;
			is_selected = 1;
		}
	}
	
	if(is_selected)
	{
		push_rect(
				  render_buffer , 
				  row.pos , row.size , 
				  GRAY_SCALE(0.75f) , 
				  NO_TRANSFORM , region
				  );
	}
	
	if(input_addr_ptr)
	{
		
		CircuitAddress input_addr = *input_addr_ptr;
		f32 height = addr_to_pixels*(input_addr.max - input_addr.min);
		f32 offset = addr_to_pixels*(input_addr.min - base_bit_addr);
		
		push_rect(
				  render_buffer , 
				  Vec2{row.pos.x , row.pos.y + offset} , 
				  Vec2{row.size.x , height} , 
				  INPUT_ADDR_ROW_COLOUR , 
				  NO_TRANSFORM , region
				  );
	}
	
	if(output_addr_ptr)
	{
		
		CircuitAddress output_addr = *output_addr_ptr;
		f32 height = addr_to_pixels*(output_addr.max - output_addr.min);
		f32 offset = addr_to_pixels*(output_addr.min - base_bit_addr);
		
		push_rect(
				  render_buffer , 
				  Vec2{row.pos.x , row.pos.y + offset} , 
				  Vec2{row.size.x , height} , 
				  OUTPUT_ADDR_ROW_COLOUR , 
				  NO_TRANSFORM , region
				  );
	}
	
	push_text(
			  render_buffer , font , 1 , 
			  addr_buffer , str_size(addr_buffer) , 
			  addr_cell.pos + Vec2{0 , addr_cell.size.y - TEXT_HEIGHT} , addr_cell.pos.x + margin_padding , 
			  TEXT_HEIGHT , 0 , TEXT_COLOUR , 
			  NO_TRANSFORM , region
			  );
	
	push_line(render_buffer , row.pos , row.pos + Vec2{row.size.x , 0} , 1.0f , TABLE_LINE_COLOUR);
	
	CircuitValue prev_val = read_circuit_value(prev_state , addr_of_interest);
	CircuitValue curr_val = read_circuit_value(curr_state , addr_of_interest);
	CircuitValue next_val = read_circuit_value(next_state , addr_of_interest);
	
	for(u32 byte_index = 0; byte_index < byte_width; byte_index++)
	{
		
		assert(byte_index < byte_width);
		//u32 curr_byte_addr = byte_addr + byte_index;
		u8 prev_value = ((u8*)&prev_val.v)[byte_index]; //((u8*)prev_mem)[curr_byte_addr];
		u8 curr_value = ((u8*)&curr_val.v)[byte_index];//((u8*)curr_mem)[curr_byte_addr];
		u8 next_value = ((u8*)&next_val.v)[byte_index];//((u8*)next_mem)[curr_byte_addr];
		
		Rect prev_value_cell = intersection(
											vertical_split(row , byte_width , byte_index) , 
											prev_value_col
											);
		
		Rect curr_value_cell = intersection(
											vertical_split(row , byte_width , byte_index) , 
											curr_value_col
											);
		
		Rect next_value_cell = intersection(
											vertical_split(row , byte_width , byte_index) , 
											next_value_col
											);
		
		str_format(prev_value_buffer , "0x%{hex8}" , prev_value);
		str_format(curr_value_buffer , "0x%{hex8}" , curr_value);
		str_format(next_value_buffer , "0x%{hex8}" , next_value);
		
		push_text(
				  render_buffer , font , 1 , 
				  prev_value_buffer , str_size(prev_value_buffer) , 
				  prev_value_cell.pos + Vec2{0 , prev_value_cell.size.y - TEXT_HEIGHT} , 
				  prev_value_cell.pos.x + margin_padding , 
				  TEXT_HEIGHT , 0 , TEXT_COLOUR , 
				  NO_TRANSFORM , region
				  );
		
		push_text(
				  render_buffer , font , 1 , 
				  curr_value_buffer , str_size(curr_value_buffer) , 
				  curr_value_cell.pos + Vec2{0 , curr_value_cell.size.y - TEXT_HEIGHT} , 
				  curr_value_cell.pos.x + margin_padding , 
				  TEXT_HEIGHT , 0 , TEXT_COLOUR , 
				  NO_TRANSFORM , region
				  );
		
		push_text(
				  render_buffer , font , 1 , 
				  next_value_buffer , str_size(next_value_buffer) , 
				  next_value_cell.pos + Vec2{0 , next_value_cell.size.y - TEXT_HEIGHT} , 
				  next_value_cell.pos.x + margin_padding , 
				  TEXT_HEIGHT , 0 , TEXT_COLOUR , 
				  NO_TRANSFORM , region
				  );
	}
	
	*next_selected_addr_ptr = next_selected_addr;
}

void do_memory_view(
					RenderBuffer * render_buffer , Font * font , InputState * input_state , 
					Rect region , f32 scroll , 
					CircuitState * prev_state , CircuitState * curr_state , CircuitState * next_state , 
					u32 mem_size , CircuitMemoryInfo * mem_info , u32 mem_info_size , 
					HardwareInstr * curr_instr , HardwareInstr * hw_instrs , u32 instr_count , 
					CircuitAddress selected_addr , CircuitAddress * next_selected_addr_ptr , u32 min_instr_index , u32 max_instr_index , 
					Mem * arena)
{
	
	void * pop_addr = mem_end(*arena , void);
	
	HardwareInstrInfo * curr_instr_info = get_hw_instr_info(curr_instr -> function);
	CircuitAddress * curr_instr_inputs = curr_instr -> inputs;
	CircuitAddress * curr_instr_outputs = curr_instr -> outputs;
	u32 curr_instr_input_count = curr_instr_info -> input_count;
	u32 curr_instr_output_count = curr_instr_info -> output_count;
	
	scroll = clip(scroll , 0 , mem_size);
	
	f32 font_width = font -> mono_width*TEXT_HEIGHT;
	f32 row_height = TEXT_HEIGHT + 5.0f;
	
	char addr_buffer[128] = {};
	Rect row_sizer = region;
	Rect col_sizer = region;
	Rect addr_col = push_right(&col_sizer , font_width*30);
	Rect prev_value_col = horizontal_split(col_sizer , 3 , 0);
	Rect curr_value_col = horizontal_split(col_sizer , 3 , 1);
	Rect next_value_col = horizontal_split(col_sizer , 3 , 2);
	
	row_sizer.pos.y += scroll*row_height;
	
	u32 min_byte_addr = 0;
	u32 max_byte_addr = mem_size;
	u32 curr_byte_addr = min_byte_addr;
	
	u32 primary_key_pressed = key_pressed(input_state -> keys[MOUSE_PRIMARY]);
	Vec2 mouse_pos = Vec2{
		input_state -> mouse.pos_x , 
		input_state -> mouse.pos_y
	};
	
	u32 * relevant_instrs = 0;
	u32 relevant_count = 0;
	
	u32 select_all = (min_instr_index == 0 && max_instr_index == instr_count && !selected_addr.min && !selected_addr.max);
	
	get_relevant_instructions(
							  selected_addr , 
							  hw_instrs , min_instr_index , max_instr_index , 
							  &relevant_instrs , &relevant_count , select_all , arena
							  );
	
	while(curr_byte_addr < max_byte_addr)
	{
		
		CircuitAddress curr_byte_circuit_addr = {curr_byte_addr*8 , curr_byte_addr*8 + 8};
		u32 min_byte_width = 1;
		addr_buffer[0] = 0;
		
		if(address_is_viewable(
							   curr_byte_circuit_addr , selected_addr , hw_instrs , 
							   relevant_instrs , relevant_count , select_all))
		{
			
			min_byte_width = MAX_U32;
			u32 found = 0;
			for(u32 info_index = 0; info_index < mem_info_size; info_index++)
			{
				
				CircuitMemoryInfo * info = mem_info + info_index;
				CircuitAddress info_addr = info -> addr;
				u32 info_byte_width = get_byte_width(info_addr);
				
				u32 bit_width = width_of(info_addr);
				u32 info_min_byte_addr = info_addr.min/8;
				
				if(info_min_byte_addr == curr_byte_addr)
				{
					
					min_byte_width = min_byte_width < info_byte_width ? min_byte_width : info_byte_width;
					char * curr_char = addr_buffer;
					curr_char += str_format(curr_char , "0x%{hex32}: +%" , info_addr.min , bit_width);
					
					if(info -> polarity == INPUT_POLARITY) curr_char += str_copy(curr_char , "in");
					if(info -> polarity == OUTPUT_POLARITY) curr_char += str_copy(curr_char , "out");
					if(info -> polarity == INOUT_POLARITY) curr_char += str_copy(curr_char , "inout");
					
					curr_char += str_copy(curr_char , " ");
					if(info -> name) curr_char += str_copy(curr_char , info -> name); 
					
					push_mem_row(
								 render_buffer , font , primary_key_pressed , mouse_pos , region , 
								 selected_addr , next_selected_addr_ptr , 
								 &row_sizer , row_height , addr_col , prev_value_col , curr_value_col , next_value_col , 
								 prev_state , curr_state , next_state , 
								 info_addr , addr_buffer , 
								 curr_instr_inputs , curr_instr_input_count ,
								 curr_instr_outputs , curr_instr_output_count
								 );
					
					found = 1;
				}
			}
			
			if(!found)
			{
				
				char * curr_char = addr_buffer;
				u32 bit_width = width_of(curr_byte_circuit_addr);
				curr_char += str_format(curr_char , "0x%{hex32}: +%" , curr_byte_circuit_addr.min , bit_width);
				min_byte_width = 1;
				
				push_mem_row(
							 render_buffer , font , primary_key_pressed , mouse_pos , region , 
							 selected_addr , next_selected_addr_ptr , 
							 &row_sizer , row_height , addr_col , prev_value_col , curr_value_col , next_value_col , 
							 prev_state , curr_state , next_state , 
							 curr_byte_circuit_addr , addr_buffer , 
							 curr_instr_inputs , curr_instr_input_count ,
							 curr_instr_outputs , curr_instr_output_count
							 );
			}
			
			assert(min_byte_width);
		}
		
		curr_byte_addr += min_byte_width;
	}
	
	curr_byte_addr = min_byte_addr;
	
	push_line(
			  render_buffer , 
			  addr_col.pos , addr_col.pos + Vec2{0 , addr_col.size.y} , 
			  1.0f , TABLE_LINE_COLOUR
			  );
	
	push_line(
			  render_buffer , 
			  curr_value_col.pos , curr_value_col.pos + Vec2{0 , curr_value_col.size.y} , 
			  1.0f , TABLE_LINE_COLOUR
			  );
	
	push_line(
			  render_buffer , 
			  next_value_col.pos , next_value_col.pos + Vec2{0 , next_value_col.size.y} , 
			  1.0f , TABLE_LINE_COLOUR
			  );
	
	
	mem_pop(arena , pop_addr);
}


struct InstrCell
{
	Rect row;
	Rect col;
	CircuitAddress address;
	HardwareInstr * instr;
	char name[64];
};

u32 write_mem_name(char * buffer , CircuitAddress addr , CircuitMemoryInfo * infos , u32 info_count)
{
	
	u32 bytes_written = 0;
	CircuitMemoryInfo * mem_info = get_circuit_mem_info(infos , info_count , addr);
	
	if(mem_info)
	{
		u32 rel_min = addr.min - mem_info -> addr.min;
		u32 rel_max = addr.max - mem_info -> addr.min;
		bytes_written = str_format(buffer , "%[%:%]" , mem_info -> name , rel_min , rel_max);
	}
	
	else
	{
		bytes_written = str_format(buffer , "0x%{hex8}" , addr.min);
	}
	
	return(bytes_written);
}

u32 is_instruction_selected(HardwareInstr * instr , 
							HardwareInstrInfo * info , 
							CircuitAddress selected_addr)
{
	
	u32 result = 0;
	
	u32 input_count = info -> input_count;
	u32 output_count = info -> output_count;
	
	CircuitAddress * inputs = instr -> inputs;
	CircuitAddress * outputs = instr -> outputs;
	
	if(instr -> function == UPDATE_IO)
	{
		result = 1;
	}
	
	else if(selected_addr.min || selected_addr.max)
	{
		if(find_addr(selected_addr , inputs , input_count))
		{
			result = 1;
		}
		
		else if (find_addr(selected_addr , outputs , output_count))
		{
			result = 1;
		}
	}
	
	else
	{
		result = 1;
	}
	
	return(result);
}


void do_instr_view(
				   RenderBuffer * render_buffer , Font * font , Rect region , f32 scroll , 
				   HardwareInstr * instrs , u32 instr_count , HardwareInstr * curr_instr , 
				   u32 * viewed_instrs , u32 viewed_instr_count , 
				   CircuitMemoryInfo * circuit_mem_info , u32 circuit_mem_info_count , 
				   CircuitAddress selected_addr , u32 min_instr_index , u32 max_instr_index , 
				   Mem * arena)
{
	
	
	HardwareInstrInfo * curr_instr_info = get_hw_instr_info(curr_instr -> function);
	void * pop_addr = mem_end(*arena , void);
	
	scroll = clip(scroll , 0 , instr_count);
	
	f32 font_width = font -> mono_width*TEXT_HEIGHT;
	f32 row_height = TEXT_HEIGHT + 5.0f;
	//f32 row_width = region.size.x;
	//f32 height_offset = region.size.y/row_height - 1;
	f32 margin_padding = 2.0f;
	u32 instr_width = font_width*16;
	u32 addr_width = font_width*20;
	
	Rect row_sizer = {
		region.pos.x , region.pos.y + region.size.y + scroll*row_height , 
		region.size.x , row_height
	};
	
	Rect col_sizer = region;
	Rect instr_col = push_right(&col_sizer , instr_width);
	Rect operand_cols[8] = {
		push_right(&col_sizer , addr_width) , 
		push_right(&col_sizer , addr_width) , 
		push_right(&col_sizer , addr_width) , 
		push_right(&col_sizer , addr_width) , 
		
		push_right(&col_sizer , addr_width) , 
		push_right(&col_sizer , addr_width) , 
		push_right(&col_sizer , addr_width) , 
		push_right(&col_sizer , addr_width)
	};
	
	u32 operand_col_capacity = array_size(operand_cols);
	InstrCell * cells = mem_end(*arena , InstrCell);
	u32 cell_count = 0;
	
	u32 max_index = viewed_instr_count;
	if(!viewed_instrs) max_index = instr_count;
	u32 view_counter = 0;
	
	u32 row_count = region.size.y / row_height + 1;
	u32 scroll_index = (u32)scroll;
	f32 scroll_offset = scroll - (f32)scroll_index;
	
	Vec2 start_of_rows = {
		region.pos.x , region.pos.y + region.size.y + scroll_offset*row_height
	};
	
	row_sizer = {
		start_of_rows.x , start_of_rows.y , 
		region.size.x , 0
	};
	
	u32 counter = 0;
	u32 start_scroll_index = scroll_index;
	u32 end_scroll_index = scroll_index + row_count;
	
	HardwareInstr * viewable_instrs[128] = {};
	u32 viewable_count = 0;
	
	for(u32 instr_index = 0; instr_index < instr_count; instr_index++)
	{
		
		HardwareInstr * instr = instrs + instr_index;
		HardwareInstrInfo * info = get_hw_instr_info(instr -> function);
		
		if(is_instruction_selected(instr , info , selected_addr))
		{
			if(start_scroll_index <= counter && counter <= end_scroll_index)
			{
				viewable_instrs[viewable_count++] = instr;
			}
			
			counter++;
		}
	}
	
	for(u32 viewable_index = 0; viewable_index < viewable_count; viewable_index++)
	{
		
		Rect row = {
			row_sizer.pos.x , row_sizer.pos.y - row_height , 
			row_sizer.size.x , row_height
		};
		
		row_sizer.pos.y -= row_height;
		row_sizer.size.y += row_height;
		
		HardwareInstr * instr = viewable_instrs[viewable_index];
		HardwareInstrInfo * info = get_hw_instr_info(instr -> function);
		
		u32 input_count = info -> input_count;
		u32 output_count = info -> output_count;
		
		CircuitAddress * inputs = instr -> inputs;
		CircuitAddress * outputs = instr -> outputs;
		
		InstrCell * cell = mem_push_struct(arena , InstrCell);
		cell -> row = row;
		cell -> col = instr_col;
		cell -> instr = instr;
		str_format(cell -> name , "%" , info -> function_name);
		u32 col_counter = 0;
		
		for(u32 input_index = 0; input_index < input_count; input_index++)
		{
			
			cell = mem_push_struct(arena , InstrCell);
			
			assert(col_counter < operand_col_capacity);
			cell -> row = row;
			cell -> col = operand_cols[col_counter++];
			cell -> instr = instr;
			cell -> address = inputs[input_index];
			
			write_mem_name(cell -> name , cell -> address , circuit_mem_info , circuit_mem_info_count);
		}
		
		for(u32 output_index = 0; output_index < output_count; output_index++)
		{
			
			cell = mem_push_struct(arena , InstrCell);
			
			assert(col_counter < operand_col_capacity);
			cell -> row = row;
			cell -> col = operand_cols[col_counter++];
			cell -> instr = instr;
			cell -> address = outputs[output_index];
			write_mem_name(cell -> name , cell -> address , circuit_mem_info , circuit_mem_info_count);
		}
		
		
	}
	
	push_clip(render_buffer , region);
	
	cell_count = mem_end(*arena , InstrCell) - cells;
	for(u32 cell_index = 0; cell_index < cell_count; cell_index++)
	{
		
		InstrCell * cell = cells + cell_index;
		HardwareInstr * instr = cell -> instr;
		Rect row = cell -> row;
		Rect col = cell -> col;
		Rect cell_region = intersection(row , col);
		char * name = cell -> name;
		
		CircuitAddress addr = cell -> address;
		
		push_clip(render_buffer , row);
		push_clip(render_buffer , col);
		
		if(instr == curr_instr)
		{
			push_rect(
					  render_buffer , row.pos , row.size , SELECTED_ROW_COLOUR , 
					  TABLE_COLOUR_LAYER , region
					  );
		}
		
		else
		{
			if(find_addr(addr , curr_instr -> inputs , curr_instr_info -> input_count))
			{
				push_rect(
						  render_buffer , cell_region.pos , cell_region.size , INPUT_ADDR_ROW_COLOUR , 
						  TABLE_COLOUR_LAYER , region
						  );
			}
			
			if(find_addr(addr , curr_instr -> outputs , curr_instr_info -> output_count))
			{
				push_rect(
						  render_buffer , cell_region.pos , cell_region.size , OUTPUT_ADDR_ROW_COLOUR , 
						  TABLE_COLOUR_LAYER , region
						  );
			}
		}
		
		Rect text_region = dilate(cell_region , -2.0f , -2.0f);
		
		push_text(
				  render_buffer , font , 1 , 
				  name , str_size(name) , 
				  text_region.pos , text_region.size.x + margin_padding , 
				  TEXT_HEIGHT , 0 , TEXT_COLOUR , 
				  TABLE_TEXT_LAYER , region
				  );
		
		pop_clip(render_buffer);
		pop_clip(render_buffer);
		
	}
	
	for(u32 row_index = 0; row_index < row_count; row_index++)
	{
		
		Vec2 start = start_of_rows - Vec2{0 , row_index*row_height};
		Vec2 end = start + Vec2{region.size.x , 0};
		
		push_line(
				  render_buffer , start , end , 
				  5.0f , TABLE_LINE_COLOUR , 
				  TABLE_OUTLINE_LAYER , NO_CLIP
				  );
	}
	
	push_line(
			  render_buffer , instr_col.pos , instr_col.pos + Vec2{0 , instr_col.size.y} , 1.0f , TABLE_LINE_COLOUR , 
			  TABLE_OUTLINE_LAYER , region
			  );
	
	for(u32 operand_index = 0; operand_index < array_size(operand_cols); operand_index++)
	{
		Rect col = operand_cols[operand_index];
		push_line(
				  render_buffer , col.pos , col.pos + Vec2{0 , col.size.y} , 1.0f , TABLE_LINE_COLOUR , 
				  TABLE_OUTLINE_LAYER , region
				  );
	}
	
	
	pop_clip(render_buffer);
	
	mem_pop(arena , pop_addr);
}


u32 write_test_op_code(char * buffer , u32 op_code)
{
	char * curr_char = buffer;
	
	if(op_code < EXPR_OP_COUNT)
	{
		curr_char += str_copy(curr_char , EXPR_OP_CODE_NAMES[op_code]);
	}
	
	else
	{
		curr_char += str_copy(curr_char , "ERROR");
	}
	
	return(curr_char-buffer);
}


u32 write_circuit_value(char * buffer , CircuitValue value)
{
	char * curr_char = buffer;
	
	curr_char += str_format(curr_char , "%:%" , value.v , value.width);
	
	return(curr_char - buffer);
}

u32 write_test_expr(
					char * buffer , TestVectorExpr * expr , 
					CircuitMemoryInfo * mem_info , u32 mem_info_count)
{
	
	char * curr_char = buffer;
	TestVectorExpr * lhs = expr -> lhs;
	TestVectorExpr * rhs = expr -> lhs;
	u32 op_code = expr -> op_code;
	
	CircuitAddress addr = expr -> addr;
	CircuitValue value = expr -> value;
	
	if(expr -> lhs && expr -> rhs)
	{
		curr_char += write_test_expr(curr_char , lhs , mem_info , mem_info_count);
		curr_char += str_copy(curr_char , "\n");
		curr_char += write_test_expr(curr_char , rhs , mem_info , mem_info_count);
		curr_char += str_copy(curr_char , "\n");
	}
	
	else if(addr.min || addr.max)
	{
		CircuitMemoryInfo * info = get_circuit_mem_info(mem_info , mem_info_count , addr);
		
		if(info)
		{
			u32 rel_min = addr.min - info -> addr.min;
			u32 rel_max = addr.max - info -> addr.min;
			curr_char += str_format(curr_char , "[%:%]%" , rel_min , rel_max , info -> name);
		}
		
		else
		{
			curr_char += str_format(curr_char , "[%:%]#x%{hex8}" , 0 , 1 , addr.min);
		}
	}
	
	else if(value.width)
	{
		curr_char += write_circuit_value(curr_char , value);
	}
	
	else
	{
		curr_char += str_copy(curr_char , "<NULL>");
	}
	
	curr_char += write_test_op_code(curr_char , op_code);
	
	return(curr_char - buffer);
}



u32 write_test_expr_result_header(char * buffer , char * start , u32 op_code , CircuitValue value)
{
	
	char * curr_char = buffer;
	
	return(curr_char-buffer);
}

u32 recursive_write_test_expr_result(
									 char * buffer , EvalExprResult * result , u32 tab , 
									 CircuitMemoryInfo * mem_info , u32 mem_info_count)
{
	
	
	char * curr_char = buffer;
	EvalExprResult * lhs = result -> lhs_result;
	EvalExprResult * rhs = result -> rhs_result;
	u32 op_code = result -> op_code;
	u32 time = result -> time;
	
	CircuitAddress addr = result -> addr;
	CircuitValue value = result -> value;
	
	curr_char += str_copy_chars(curr_char , '\t' , tab);
	
	if(lhs && rhs)
	{
		
		curr_char += write_test_op_code(curr_char , op_code);
		curr_char += str_copy(curr_char , "  ");
		curr_char += str_tabulate(curr_char , buffer , 50 , 4);
		curr_char += write_circuit_value(curr_char , value);
		curr_char += str_format(curr_char , " t=%\n" , time);
		
		curr_char += recursive_write_test_expr_result(curr_char , lhs , tab+1 , mem_info , mem_info_count);
		curr_char += recursive_write_test_expr_result(curr_char , rhs , tab+1 , mem_info , mem_info_count);
	}
	
	else
	{
		curr_char += write_test_op_code(curr_char , op_code);
		curr_char += str_copy(curr_char , " ");
		
		if(addr.min || addr.max)
		{
			CircuitMemoryInfo * info = get_circuit_mem_info(mem_info , mem_info_count , addr);
			
			if(info)
			{
				u32 rel_min = addr.min - info -> addr.min;
				u32 rel_max = addr.max - info -> addr.min;
				curr_char += str_format(curr_char , "[%:%]%" , rel_min , rel_max , info -> name);
			}
			
			else
			{
				curr_char += str_format(curr_char , "[%:%]#x%{hex8}" , 0 , 1 , addr.min);
			}
		}
		
		else if(value.width)
		{
			curr_char += write_circuit_value(curr_char , value);
		}
		
		else
		{
			curr_char += str_copy(curr_char , "<NULL>");
		}
		
		curr_char += str_tabulate(curr_char , buffer , 50 , 4);
		curr_char += write_circuit_value(curr_char , value);
		curr_char += str_format(curr_char , " t=%\n" , time);
	}
	
	return(curr_char - buffer);
}

void push_test_expr(
					RenderBuffer * render_buffer , Font * font , Rect region , 
					TestVectorExpr * expr , CircuitMemoryInfo * mem_info , u32 mem_info_count , Mem * arena)
{
	
	char * buffer = mem_end(*arena , char);
	u32 size = write_test_expr(
							   buffer , expr , 
							   mem_info , mem_info_count
							   );
	
	push_text(
			  render_buffer , font , 1 , 
			  buffer , size , 
			  region.pos , region.pos.x , 
			  TEXT_HEIGHT , 0 , TEXT_COLOUR , 
			  NO_TRANSFORM , region
			  );
}

void do_test_overview(
					  RenderBuffer * render_buffer , Font * font , Rect region , f32 scroll , 
					  TestVector * tests , u32 test_count , CircuitMemoryInfo * mem_info , u32 mem_info_count , 
					  Mem * arena)
{
	
	
	f32 row_height = TEXT_HEIGHT;
	Rect conseq_col = horizontal_split(region , 2 , 0);
	Rect antec_col = horizontal_split(region , 2 , 1);
	
	for(u32 test_index = 0; test_index < test_count; test_index++)
	{
		
		TestVector * curr_test = tests + test_index;
		TestVectorExpr ** conseq_array = curr_test -> conseq;
		TestVectorExpr ** antec_array = curr_test -> antec;
		
		u32 conseq_count = curr_test -> conseq_count;
		u32 antec_count = curr_test -> antec_count;
		
		u32 row_count = max(conseq_count , antec_count);
		Rect table_chunk = push_down(&region , row_count*row_height);
		
		for(u32 cond_index = 0; cond_index < row_count; cond_index++)
		{
			
			Rect row = vertical_split(table_chunk , row_count , row_count - (cond_index+1));
			Rect conseq_cell = intersection(row , conseq_col);
			Rect antec_cell = intersection(row , antec_col);
			
			if(cond_index < conseq_count)
			{
				TestVectorExpr * conseq = conseq_array[cond_index];
				push_test_expr(render_buffer , font , conseq_cell , conseq , mem_info , mem_info_count , arena);
			}
			
			if(cond_index < antec_count)
			{
				TestVectorExpr * antec = antec_array[cond_index];
				push_test_expr(render_buffer , font , antec_cell , antec , mem_info , mem_info_count , arena);
			}
		}
		
		push_line(render_buffer , table_chunk.pos , table_chunk.pos + Vec2{table_chunk.size.x , 0} , 1.0f,  {0,0,0,1});
	}
}



void add_to_operands(
					 u32 new_addr , u32 new_value , 
					 u32 * addrs , u32 * values , u32 * size_ptr , u32 capacity)
{
	u32 size = *size_ptr;
	u32 found = 0;
	for(u32 i = 0; i < size; i++)
	{
		if(addrs[i] == new_addr)
		{
			found = 1;
			break;
		}
		
		
	}
	
	if(!found)
	{
		assert(size < capacity);
		addrs[size] = new_addr;
		values[size] = new_value;
		size++;
	}
	
	*size_ptr = size;
}


void push_right_edge(RenderBuffer * render_buffer , Rect rect , f32 thickness , Vec4 colour , OPTIONAL_PUSH_ARGS)
{
	
	push_line(
			  render_buffer , rect.pos + Vec2{rect.size.x , 0} , rect.pos + rect.size , thickness , colour , 
			  OPTIONAL_PUSH_ARG_NAMES
			  );
	
}


void show_test_results(
					   RenderBuffer * render_buffer , Font * font , Rect region , 
					   TestVectorResult * results , u32 result_count , 
					   CircuitMemoryInfo * mem_info , u32 info_count , Mem * arena)
{
	
	char * buffer = mem_end(*arena , char);
	char * curr_char = buffer;
	
	//f32 font_height = TEXT_HEIGHT;
	//f32 font_width = font -> mono_width*font_height;
	
	//Rect sizer = region;
	//Rect status_col = push_right(&sizer , font_width*6);
	//Rect conseq_col = push_right(&sizer , font_width*20);
	//Rect antec_col = push_right(&sizer , font_width*20);
	
	//char * passed_text = "PASSED";
	//char * failed_text = "FAILED";
	
	for(u32 test_index = 0; test_index < result_count; test_index++)
	{
		TestVectorResult * test_result = results + test_index;
		
		EvalExprResult ** conseq_results = test_result -> conseq_results;
		EvalExprResult ** antec_results = test_result -> antec_results;
		u32 conseq_count = test_result -> conseq_count;
		u32 antec_count = test_result -> antec_count;
		u32 passed = test_result -> passed;
		
		if(test_result -> active)
		{
			
			curr_char += str_format(curr_char , "%\n" , passed ? "PASSED" : "FAILED");
			
			for(u32 conseq_index = 0; conseq_index < conseq_count; conseq_index++)
			{
				
				if(conseq_index == 0) curr_char += str_copy(curr_char , "conseq\n");
				EvalExprResult * cond_result = conseq_results[conseq_index];
				
				if(!cond_result -> value.v)
				{
					curr_char += str_copy(curr_char , "***\n");
				}
				
				curr_char += recursive_write_test_expr_result(curr_char , cond_result , 0 , mem_info , info_count);
				curr_char += str_copy(curr_char , "\n");
				//curr_char += str_format(curr_char , "${tabulate:%}" , tab_amount);
			}
			
			for(u32 antec_index = 0; antec_index < antec_count; antec_index++)
			{
				if(antec_index == 0) curr_char += str_copy(curr_char , "antec\n");
				
				EvalExprResult * cond_result = antec_results[antec_index];
				
				curr_char += recursive_write_test_expr_result(curr_char , cond_result , 0 , mem_info , info_count);
				curr_char += str_copy(curr_char , "\n");
			}
		}
	}
	
	//push_right_edge(render_buffer , status_col , 1.0f , {0,0,0,1});
	//push_right_edge(render_buffer , conseq_col , 1.0f , {0,0,0,1});
	//push_right_edge(render_buffer , antec_col , 1.0f , {0,0,0,1});
	
	Vec2 start_pos = region.pos + Vec2{0 , region.size.y - TEXT_HEIGHT};
	push_hollow_rect(render_buffer , region.pos , region.size , 1.0f , {0,0,0,1});
	
	push_text(
			  render_buffer , font , 1 , 
			  buffer , curr_char - buffer , 
			  start_pos , start_pos.x , 
			  TEXT_HEIGHT , 0 , TEXT_COLOUR
			  );
}



void get_module_children(
						 u32 parent_index , 
						 u32 * modules , u32 * module_count_ptr , u32 module_capacity , 
						 CircuitModuleLink * links , u32 link_count)
{
	
	u32 module_count = 0;
	
	for(u32 link_index = 0; link_index < link_count; link_index++)
	{
		CircuitModuleLink * curr_link = links + link_index;
		
		if(curr_link -> parent_index == parent_index)
		{
			assert(module_count < module_capacity);
			modules[module_count++] = curr_link -> child_index;
		}
	}
	
	*module_count_ptr = module_count;
}

CircuitModule * get_parent(CircuitModule * child_module , CircuitModule * modules , CircuitModuleLink * links , u32 link_count)
{
	
	CircuitModule * result = 0;
	u32 child_index = child_module-modules;
	
	for(u32 link_index = 0; link_index < link_count; link_index++)
	{
		CircuitModuleLink * curr_link = links + link_index;
		
		if(curr_link -> child_index == child_index)
		{
			result = modules + curr_link -> parent_index;
			break;
		}
	}
	
	return(result);
}


struct ModuleRow
{
	CircuitModule * module;
	char * name;
	Rect region;
};

void do_component_view(
					   RenderBuffer * render_buffer , Font * font , InputState * input_state , Mem * arena , Rect region , 
					   CircuitModule * modules , u32 module_count , 
					   CircuitModuleLink * links , u32 link_count , 
					   u32 * selected_module_index_ptr)
{
	
	u32 selected_module_index = *selected_module_index_ptr;
	CircuitModule * selected_module = 0;
	CircuitModule * parent_module = 0;
	if(selected_module_index < module_count)
	{
		selected_module = modules + selected_module_index;
		parent_module = get_parent(selected_module , modules , links , link_count);
	}
	
	Rect row_sizer = region;
	Rect name_col = region;
	f32 row_height = TEXT_HEIGHT;
	
	u32 primary_pressed = key_pressed(input_state -> keys[MOUSE_PRIMARY]);
	Vec2 mouse_pos = {input_state -> mouse.pos_x , input_state -> mouse.pos_y};
	
	ModuleRow * module_rows = mem_end(*arena , ModuleRow);
	
	ModuleRow * back_row = mem_push_struct(arena , ModuleRow);
	back_row -> name = "back";
	back_row -> module = parent_module;
	back_row -> region = push_down(&row_sizer , row_height);
	
	for(u32 i = 0; i < module_count; i++)
	{
		
		CircuitModule * curr_module = modules + i;
		CircuitModule * parent = get_parent(curr_module , modules , links , link_count);
		if(parent == selected_module)
		{
			
			ModuleRow * row = mem_push_struct(arena , ModuleRow);
			row -> name = curr_module -> name;
			row -> module = curr_module;
			row -> region = push_down(&row_sizer , row_height);
		}
	}
	
	u32 row_count = mem_end(*arena , ModuleRow) - module_rows;
	
	for(u32 row_index = 0; row_index < row_count; row_index++)
	{
		
		ModuleRow * row = module_rows + row_index;
		CircuitModule * module = row -> module;
		char * name = row -> name;
		Rect row_region = row -> region;
		Rect name_cell = intersection(row_region , name_col);
		
		push_text(
				  render_buffer , font , 1 , 
				  name , str_size(name) , 
				  name_cell.pos , name_cell.pos.x , 
				  TEXT_HEIGHT , 0 , TEXT_COLOUR
				  );
		
		if(primary_pressed)
		{
			if(in_rect(mouse_pos , row_region))
			{
				selected_module_index = module - modules;
			}
		}
	}
	
	push_hollow_rect(render_buffer , region.pos , region.size , 1.0f , {0,0,0,1});
	
	*selected_module_index_ptr = selected_module_index;
}


struct TraceElement
{
	
	CircuitEvent event;
	Rect region;
	u32 makes_loop;
};

void do_signal_backtrace_view(
							  RenderBuffer * render_buffer , Font * font , InputState * input_state , Rect region , 
							  EventBuffer * event_buffer , CircuitAddress start_addr , CircuitValue start_addr_value , 
							  CircuitAddress * next_selected_addr_ptr , i32 time , 
							  Vec2 * view_offset_ptr , f32 * view_scale_ptr , 
							  CircuitMemoryInfo * mem_info , u32 mem_info_count , Mem * arena)
{
	
	CircuitAddress next_selected_addr = *next_selected_addr_ptr;
	Vec2 view_offset = *view_offset_ptr;
	f32 view_scale = *view_scale_ptr;
	
	f32 speed = 10.0f;
	f32 scroll = input_state -> mouse.scroll_wheel;
	u32 primary_key_pressed = key_pressed(input_state -> keys[MOUSE_PRIMARY]);
	f32 left_held = (key_held(input_state -> keys[KEY_A]) > 0)*speed;
	f32 right_held = (key_held(input_state -> keys[KEY_D]) > 0)*speed;
	f32 up_held = (key_held(input_state -> keys[KEY_W]) > 0)*speed;
	f32 down_held = (key_held(input_state -> keys[KEY_S]) > 0)*speed;
	
	Vec2 mouse_pos = Vec2{
		input_state -> mouse.pos_x , 
		input_state -> mouse.pos_y
	};
	
	Vec2 mouse_rel_pos = view_offset + (mouse_pos - region.pos)/view_scale;
	
	//view_offset -= mouse_rel_pos;
	//view_offset -= zoom_delta*view_offset;
	//view_offset += mouse_rel_pos;
	
	f32 zoom_delta = scroll*view_scale*0.01f;
	f32 new_view_scale = clip(zoom_delta + view_scale , 0.00001 , 10000);
	f32 mouse_delta = zoom_delta/(view_scale*new_view_scale);
	
	view_offset += Vec2{
		(right_held - left_held)/view_scale , 
		(up_held - down_held)/view_scale
	};
	
	view_scale = new_view_scale;
	view_offset += mouse_rel_pos*mouse_delta;
	//view_offset += mouse_rel_pos;
	//view_offset += (view_offset-mouse_rel_pos)*zoom_delta*view_scale;
	
	
	Vec2 start_min = (-view_offset)*view_scale;
	Vec2 start_max = (-view_offset + region.size)*view_scale;
	Rect start_region = {start_min + region.pos , start_max-start_min};
	
	//start_region.pos -= view_offset;
	//start_region.size *= view_scale;
	
	CircuitEvent * events = event_buffer -> events;
	u32 start_index = event_buffer -> event_start_index;
	u32 end_index = event_buffer -> event_end_index;
	u32 capacity = event_buffer -> event_capacity;
	u32 index = end_index;
	
	CircuitEvent * next_event = 0;
	CircuitEvent * first_event = 0;
	
	TraceElement backtrace[1024] = {};
	u32 backtrace_capacity = array_size(backtrace);
	u32 backtrace_size = 0;
	
	TraceElement stack[512] = {};
	u32 stack_capacity = array_size(stack);
	u32 stack_size = 1;
	
	CircuitEvent drivers[8] = {};
	u32 driver_capacity = array_size(drivers);
	u32 driver_count = 1;
	
	//CircuitEvent * recent_event = find_most_recent_signal_event();
	
	stack[0].event.type = CIRCUIT_SIGNAL_EVENT;
	stack[0].event.source = start_addr;
	stack[0].event.time = time;
	stack[0].event.result_value = start_addr_value;
	stack[0].event.input_value = start_addr_value;
	stack[0].region = start_region;
	
	while(stack_size)
	{
		
		stack_size--;
		assert(backtrace_size < backtrace_capacity);
		backtrace[backtrace_size++] = stack[stack_size];
		
		TraceElement curr_ele = stack[stack_size];
		CircuitEvent curr_event = stack[stack_size].event;
		assert(curr_event.type == CIRCUIT_SIGNAL_EVENT);
		
		driver_count = 0;
		first_event = 0;
		next_event = 0;
		index = end_index;
		
		while(index != start_index)
		{
			
			CircuitEvent * event = events + index;
			i32 event_time = event -> time;
			
			if(event -> type == CIRCUIT_SIGNAL_EVENT)
			{
				if(addr_overlap(event -> dest , curr_event.source) && event_time < curr_event.time)
				{
					
					u32 found = 0;
					i32 max_time = 0;
					
					for(u32 driver_index = 0; driver_index < driver_count; driver_index++)
					{
						
						u32 driver_time = drivers[driver_index].time;
						max_time = driver_time > max_time ? driver_time : max_time;
						
						// NOTE: cuts loops
						if(drivers[driver_index].source == event -> source)
						{
							found = 1;
							break;
						}
					}
					
					if(!found && event_time >= max_time)
					{
						assert(driver_count < driver_capacity);
						drivers[driver_count] = *event;
						driver_count++;
					}
				}
			}
			
			if(index-- == 0) index = capacity-1;
		}
		
		push_down(&curr_ele.region , TEXT_HEIGHT);
		
		for(u32 driver_index = 0; driver_index < driver_count; driver_index++)
		{
			
			
			CircuitEvent event = drivers[driver_index];
			u32 found = 0;
			
			for(u32 backtrace_index = 0; backtrace_index < backtrace_size; backtrace_index++)
			{
				if(backtrace[backtrace_index].event.source == event.source)
				{
					found = 1;
					break;
				}
			}
			
			TraceElement * ele = 0;
			if(found)
			{
				// NOTE: cuts loops by skipping the stack.
				assert(backtrace_size < backtrace_capacity);
				ele = backtrace + backtrace_size++;
				if(event.source == start_addr)
				{
					ele -> makes_loop = 1;
				}
			}
			
			else
			{
				assert(stack_size < stack_capacity);
				ele = stack + stack_size++;
			}
			
			ele -> event = event;
			ele -> region = horizontal_split(curr_ele.region , driver_count , driver_index);
		}
	}
	
	push_hollow_rect(render_buffer , region.pos , region.size , 1.0 , {0,0,0,1});
	push_clip(render_buffer , region);
	
#if 0
	push_transform(
				   render_buffer , 
				   push_translation(
									scale_mat4({view_scale , view_scale , 1}) , 
									Vec3{view_offset.x , view_offset.y , 0}
									)
				   );
	
#endif
	
	for(u32 trace_index = 0; trace_index < backtrace_size; trace_index++)
	{
		
		TraceElement * trace = backtrace + trace_index;
		Rect trace_region = trace -> region;
		
		//region.pos -= view_offset;
		//region.size *= view_scale;
		trace -> region = trace_region;
	}
	
	for(u32 trace_index = 0; trace_index < backtrace_size; trace_index++)
	{
		
		TraceElement trace = backtrace[trace_index];
		CircuitAddress trace_src_addr = trace.event.source;
		CircuitAddress trace_dest_addr = trace.event.dest;
		//CircuitValue trace_result_value = trace.event.result_value;
		CircuitValue trace_input_value = trace.event.input_value;
		i32 trace_time = trace.event.time;
		u32 instr_func = trace.event.instr_func;
		u32 makes_loop = trace.makes_loop;
		Rect trace_region = trace.region;
		
		HardwareInstrInfo * instr_info = get_hw_instr_info(instr_func);
		assert(instr_info);
		
		char * curr_char = 0;
		char * left_buffer = mem_push_array(arena , char , 128);
		char * right_buffer = mem_push_array(arena , char , 128);
		
		curr_char = left_buffer;
		curr_char += str_format(curr_char , "t=%, " , trace_time);
		curr_char += write_mem_name(curr_char , trace_src_addr , mem_info , mem_info_count);
		curr_char += str_format(curr_char , " % " , instr_info -> function_name);
		curr_char += write_circuit_value(curr_char , trace_input_value);
		
		curr_char = right_buffer;
		curr_char += write_mem_name(curr_char , trace_dest_addr , mem_info , mem_info_count);
		right_buffer[0] = 0;
		
		f32 right_buffer_width = get_text_width(right_buffer , TEXT_HEIGHT , font) + 10;
		Vec2 left_pos = trace_region.pos + Vec2{0 , trace_region.size.y - TEXT_HEIGHT};
		Vec2 right_pos = trace_region.pos + Vec2{trace_region.size.x - right_buffer_width , trace_region.size.y - TEXT_HEIGHT};
		
		Rect text_region = {left_pos , {right_pos.x-left_pos.x , TEXT_HEIGHT}};
		
		if(makes_loop)
		{
			push_rect(render_buffer , text_region.pos , text_region.size , {1,0,0,1});
		}
		
		push_text(
				  render_buffer , font , 1 , 
				  left_buffer , str_size(left_buffer) , 
				  left_pos, left_pos.x ,
				  TEXT_HEIGHT , 0 , TEXT_COLOUR , 
				  NO_TRANSFORM , text_region
				  );
		
		push_text(
				  render_buffer , font , 1 , 
				  right_buffer , str_size(right_buffer) , 
				  right_pos , right_pos.x , 
				  TEXT_HEIGHT , 0 , TEXT_COLOUR , 
				  NO_TRANSFORM , text_region
				  );
		
		push_hollow_rect(
						 render_buffer , 
						 trace_region.pos , trace_region.size , 1.0 , {0,0,0,1}
						 );
		
		if(in_rect(mouse_pos , trace_region) && primary_key_pressed)
		{
			if(trace_src_addr == start_addr)
			{
				next_selected_addr = {};
			}
			
			else
			{
				next_selected_addr = trace_src_addr;
			}
		}
	}
	
	//pop_transform(render_buffer);
	pop_clip(render_buffer);
	
	*next_selected_addr_ptr = next_selected_addr;
	*view_offset_ptr = view_offset;
	*view_scale_ptr = view_scale;
}



void do_driver_list_view(
						 RenderBuffer * render_buffer , Font * font , InputState * input_state , Rect region , 
						 CircuitAddress selected_addr , CircuitAddress * next_selected_addr_ptr , i32 time , 
						 EventBuffer * event_buffer , CircuitMemoryInfo * mem_info , u32 mem_info_count)
{
	
	u32 primary_key_pressed = key_pressed(input_state -> keys[MOUSE_PRIMARY]);
	Vec2 mouse_pos = Vec2{
		input_state -> mouse.pos_x , 
		input_state -> mouse.pos_y
	};
	
	CircuitAddress next_selected_addr = *next_selected_addr_ptr;
	
	CircuitEvent * events = event_buffer -> events;
	u32 start_index = event_buffer -> event_start_index;
	u32 end_index = event_buffer -> event_end_index;
	u32 capacity = event_buffer -> event_capacity;
	u32 index = end_index;
	
	push_clip(render_buffer , region);
	
	u32 first_found = 0;
	i32 target_time = 0;
	
	while(index != start_index)
	{
		
		CircuitEvent * event = events + index;
		i32 event_time = event -> time;
		
		if(event -> type == CIRCUIT_SIGNAL_EVENT)
		{
			if(addr_overlap(event -> source , selected_addr))
			{
				
				if(!first_found && event_time <= time)
				{
					target_time = event_time;
					first_found = 1;
				}
				
				if(event_time == target_time)
				{
					
					Rect row = push_down(&region , TEXT_HEIGHT);
					if(in_rect(mouse_pos , row) && primary_key_pressed)
					{
						next_selected_addr = event -> dest;
					}
					
					char buffer[128] = {};
					char * curr_char = buffer;
					
					curr_char += str_format(curr_char , "t=% , " , event_time);
					curr_char += write_mem_name(curr_char , event -> dest , mem_info , mem_info_count);
					push_hollow_rect(render_buffer , row.pos , row.size , 1.0f , {0,0,0,1});
					
					push_text(
							  render_buffer , font , 1 , 
							  buffer , str_size(buffer) , 
							  row.pos , row.pos.x ,
							  TEXT_HEIGHT , 0 , TEXT_COLOUR
							  );
				}
				
				else
				{
					//break;
				}
			}
		}
		
		if(index-- == 0) index = capacity-1;
	}
	
	pop_clip(render_buffer);
	
	
	*next_selected_addr_ptr = next_selected_addr;
	
}




void do_completed_instr_view(
							 RenderBuffer * render_buffer , Font * font , Rect region , 
							 InstrTracker * tracker , Instr * sw_instrs , MipsSpec * spec)
{
	
	//CircuitAddress PC_addr = mips -> PC_addr;
	//InstrStage * stage_list = tracker -> stage_list;
	CompletedInstr * completed_instrs = tracker -> completed_instrs;
	u32 completed_instr_count = tracker -> completed_instr_count;
	char buffer[128] = {};
	
	for(u32 completed_index = 0; completed_index < completed_instr_count; completed_index++)
	{
		
		CompletedInstr * completed_instr = completed_instrs + completed_index;
		u32 pc_value = completed_instr -> pc_value;
		Instr * instr = sw_instrs + pc_value;
		
		Rect row = push_down(&region ,  TEXT_HEIGHT);
		char * curr_char = buffer;
		
		curr_char += str_format(curr_char , "PC = % , " , pc_value);
		curr_char += write_instruction(curr_char , instr , spec -> lines , spec -> instr_set , spec -> fields , spec -> regs);
		
		push_text(
				  render_buffer , font , 1 , 
				  buffer , str_size(buffer) , 
				  row.pos , row.pos.x , 
				  TEXT_HEIGHT , 0 , TEXT_COLOUR
				  );
	}
}


void do_assembly_view(
					  RenderBuffer * render_buffer , Font * font , Rect region , 
					  Instr * instrs , u32 instr_count , MipsSpec * spec , 
					  CircuitState * curr_state , Mem * arena)
{
	
	Mem lines = spec -> lines;
	Mem reg_arena = spec -> regs;
	Mem instr_set_arena = spec -> instr_set;
	Mem instr_field_arena = spec -> fields;
	
	char * buffer = mem_end(*arena , char);
	CircuitAddress PC_addr = get_line_addr(lines , PC_LINE);
	CircuitValue PC_value = read_circuit_value(curr_state , PC_addr);
	
	u32 pc_instr_index = 0 + cv_to_int(PC_value , 0);
	
	for(u32 instr_index = 0; instr_index < instr_count; instr_index++)
	{
		
		Instr * curr_instr = instrs + instr_index;
		char * curr_char = buffer;
		Rect line_region = push_down(&region , TEXT_HEIGHT);
		
		curr_char[0] = 0;
		curr_char += write_instruction(curr_char , curr_instr , lines , instr_set_arena , instr_field_arena , reg_arena);
		
		
		if(instr_index == pc_instr_index)
		{
			push_hollow_rect(render_buffer , line_region.pos , line_region.size , 1.0f , {1,0,0,1});
		}
		
		push_text(
				  render_buffer , font , 1 , 
				  buffer , curr_char-buffer , 
				  line_region.pos , line_region.pos.x , 
				  TEXT_HEIGHT , 0 , TEXT_COLOUR , 
				  NO_TRANSFORM , line_region
				  );
	}
}



struct WatchPoint
{
	char name[64];
	CircuitAddress addr;
	CircuitValue value;
};

WatchPoint * push_watch(
						char * name , CircuitAddress addr , CircuitState * curr_state , Mem * arena)
{
	WatchPoint * point = mem_push_struct(arena , WatchPoint);
	
	str_copy(point -> name , name);
	point -> addr = addr;
	point -> value = read_circuit_value(curr_state , addr);
	
	return(point);
	
}



void do_watch_list_view(
						RenderBuffer * render_buffer , Font * font , Rect region , 
						WatchPoint * watch_list , u32 count)
{
	
	push_hollow_rect(render_buffer , region.pos , region.size , 1.0f , {0,0,0,1});
	
	char value_buffer[128] = {};
	Rect name_col = horizontal_split(region , 2 , 0);
	Rect value_col = horizontal_split(region , 2 , 1);
	
	push_clip(render_buffer , region);
	
	for(u32 row_index = 0; row_index < count; row_index++)
	{
		
		CircuitValue value = watch_list[row_index].value;
		char * name = watch_list[row_index].name;
		
		str_format(value_buffer , "%" , cv_to_int(value , 0));
		Rect row = push_down(&region , TEXT_HEIGHT);
		Rect name_cell = intersection(row , name_col);
		Rect value_cell = intersection(row , value_col);
		
		push_text(
				  render_buffer , font , 1 , 
				  name , str_size(name) , 
				  name_cell.pos , name_cell.pos.x , 
				  TEXT_HEIGHT , 0 , TEXT_COLOUR , 
				  NO_TRANSFORM , name_cell
				  );
		
		push_text(
				  render_buffer , font , 1 , 
				  value_buffer , str_size(value_buffer) , 
				  value_cell.pos , value_cell.pos.x , 
				  TEXT_HEIGHT , 0 , TEXT_COLOUR , 
				  NO_TRANSFORM , value_cell
				  );
		
		push_hollow_rect(render_buffer , row.pos , {row.size.x , 1.0f} , 1.0f,  {0,0,0,1});
		
	}
	
	pop_clip(render_buffer);
}

void do_line_display(
					 RenderBuffer * render_buffer , Font * font , Rect region , 
					 CircuitMemoryInfo * mem_info , u32 mem_info_count , 
					 Mem line_arena , CircuitState * curr_state , Mem * arena)
{
	
	char buffer[128] = {};
	
	LineSpec * lines = mem_start(line_arena , LineSpec);
	u32 line_count = mem_size(line_arena , LineSpec);
	WatchPoint * watch_list = mem_end(*arena , WatchPoint);
	
	for(u32 line_index = 0; line_index < line_count; line_index++)
	{
		
		LineSpec * curr_line = lines + line_index;
		char * name = curr_line -> name;
		CircuitAddress addr = curr_line -> addr;
		
		char * curr_char = buffer;
		
		curr_char += str_format(curr_char , "% " , name);
		curr_char += write_mem_name(curr_char , addr , mem_info , mem_info_count);
		
		push_watch(buffer , addr , curr_state , arena);
	}
	
	u32 watch_count = mem_end(*arena , WatchPoint) - watch_list;
	
	do_watch_list_view(
					   render_buffer , font , region , 
					   watch_list , watch_count
					   );	
	
	
}


void do_console_view(
					 RenderBuffer * render_buffer , Font * font , Rect region , 
					 char * text , u32 text_size)
{
	
	u32 line_count = str_line_count(text , text_size);
	Vec2 start_pos = region.pos + Vec2{0 , TEXT_HEIGHT*(line_count+1)};
	
	push_text(
			  render_buffer , font , 1 , 
			  text , text_size , 
			  start_pos , start_pos.x , 
			  TEXT_HEIGHT , 0 , TEXT_COLOUR , 
			  NO_TRANSFORM , region
			  );
}




void stats_to_console(StatTracker * stats)
{
	
#if 0
	StatFrame * frames = stats -> frames;
	u32 frame_count = stats -> frame_count;
	//char buffer[128] = {};
	
	for(u32 frame_index = 0; frame_index < frame_count; frame_index++)
	{
		
		StatFrame * frame = frames + frame_index;
		i32 time = frame -> time;
		u32 total_power = frame -> total_power;
		u32 total_switches = frame -> total_switches;
		//char * curr_char = buffer;
		//curr_char += str_format(curr_char , "");
		
		output_to_console("[%]${tabulate:4}power = %${tabulate:8} switches = %\n" , 0 , time , total_power , total_switches);
	}
#else
	
	u32 total_energy = stats -> total_energy;
	u32 total_switches = stats -> total_switches;
	u32 total_frames = stats -> frame_count;
	u32 max_energy = stats -> max_energy;
	u32 max_switches = stats -> max_switches;
	
	u32 avg_energy = total_frames ? total_energy / total_frames : 0;
	u32 avg_switches = total_frames ? total_switches / total_frames : 0;
	
	
	output_to_console(
					  "(avg,max) energy = (%,%)${tabulate:8}"
					  "(avg,max) switches = (%,%)\n" , 0 , 
					  avg_energy ,  max_energy , 
					  avg_switches , max_switches
					  );
	
#endif
}

void do_stats_graph(RenderBuffer * render_buffer , Font * font , Rect region , StatTracker * stats)
{
	
	
	
	StatFrame * frames = stats -> frames;
	u32 max_energy = stats -> max_energy;
	u32 frame_count = stats -> frame_count;
	u32 y_range = max_energy;
	u32 x_range = frame_count;
	
	char * x_caption = "time (frames)";
	char * y_caption = "energy";
	
	u32 x_cap_size = str_size(x_caption);
	u32 y_cap_size = str_size(y_caption);
	
	Vec2 x_cap_pos = region.pos + Vec2{region.size.x*0.5f , 0};
	Vec2 y_cap_pos = region.pos + Vec2{0 , region.size.y*0.5f};
	
	Rect x_cap_region = get_text_bounds(x_caption , x_cap_size , 
										x_cap_pos , TEXT_HEIGHT , {} , font);
	
	Rect y_cap_region = get_text_bounds(y_caption , y_cap_size , 
										y_cap_pos , TEXT_HEIGHT , {} , font);
	
	push_text(render_buffer , font , 1 , 
			  x_caption , x_cap_size , 
			  x_cap_pos , 0 , 
			  TEXT_HEIGHT , 0 , {0,0,0,1});
	
	
	push_text(render_buffer , font , 1 , 
			  y_caption , y_cap_size , 
			  y_cap_pos , 0 , 
			  TEXT_HEIGHT , 0 , {0,0,0,1});
	
	Rect graph_region = dilate(region , -(10 + y_cap_region.size.x) , -(10 + x_cap_region.size.y));
	Vec2 origin = graph_region.pos;
	Vec2 axis_lengths = graph_region.size;
	Vec2 x_end = graph_region.pos + Vec2{axis_lengths.x , 0};
	Vec2 y_end = graph_region.pos + Vec2{0 , axis_lengths.y};
	
	push_hollow_rect(render_buffer , region.pos , region.size , 1.0f , {0,0,0,1});
	//push_hollow_rect(render_buffer , graph_region.pos , graph_region.size , 1.0f , {0,0,0,1});
	
	push_line(render_buffer , origin , x_end , 1.0f , {0,0,0,1});
	push_line(render_buffer , origin , y_end , 1.0f , {0,0,0,1});
	
	
	for(u32 frame_index = 0; frame_index+1 < frame_count; frame_index++)
	{
		
		u32 curr_index = frame_index + 0;
		u32 next_index = frame_index + 1;
		
		StatFrame * curr_frame = frames + curr_index;
		StatFrame * next_frame = frames + next_index;
		
		u32 curr_energy = curr_frame -> norm_static_energy + curr_frame -> norm_switch_energy;
		u32 next_energy = next_frame -> norm_static_energy + next_frame -> norm_switch_energy;
		
		Vec2 curr_uv = Vec2{(f32)curr_index/(f32)x_range , (f32)curr_energy/(f32)y_range};
		Vec2 next_uv = Vec2{(f32)next_index/(f32)x_range , (f32)next_energy/(f32)y_range};
		
		Vec2 curr_pos = origin + axis_lengths*curr_uv;
		Vec2 next_pos = origin + axis_lengths*next_uv;
		
		push_line(render_buffer , curr_pos , next_pos , 1.0 , {curr_uv.x,1.0f-curr_uv.x,0,1});
		
	}
}



void timers_to_console(DebugTimer * timers , u32 count)
{
	output_to_console("--- debug timers ---\n" , 0);
	for(u32 i = 0; i < count; i++)
	{
		DebugTimer timer = timers[i];
		u64 time = timer.end - timer.start;
		output_to_console("% = ${tabulate:20}%\n" , 0 , timer.name , time);
	}
	
	output_to_console("---              ---\n" , 0);
}


void addr_power_list_to_console(CircuitAddressPower * power_list , u32 count , CircuitMemoryInfo * mem_info , u32 info_count)
{
	
	output_to_console("--- switch power ---\n" , 0);
	for(u32 power_index = 0; power_index < count; power_index++)
	{
		
		CircuitAddressPower * power = power_list + power_index;
		CircuitAddress addr = power -> addr;
		u32 energy = get_norm_switch_energy(*power);
		
		if(energy)
		{
			char name[64] = {};
			write_mem_name(name , addr , mem_info , info_count);
			output_to_console("%${tabulate:23}%\n" , 0 , name , energy);
		}
	}
	output_to_console("---              ---\n" , 0);
	
}

